/* 
 * Copyright 1999,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.log4j.appender;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 * Responsible for encapsulating time-oriented operations on log filenames or
 * partial log filenames via the concatenation methods.
 * 
 * @author <a href="mailto:simon_park_mail AT yahoo DOT co DOT uk">Simon Park</a>
 * @version 1.8
 */
final class TimeSuffixHelper {

  private final Calendar calendar;

  private final DateFormat dateFormat;

  /**
   * Initialise to format log filenames and partial filenames using the current
   * default {@link java.util.Locale} and {@link java.util.TimeZone}.
   */
  TimeSuffixHelper(final AppenderRollingProperties properties) {
    super();
    /*
     * Locale is only important for formatting date patterns, otherwise always
     * use default.
     */
    this.calendar = Calendar.getInstance();
    DateFormat format = DateFormat.getDateTimeInstance(DateFormat.DEFAULT,
        DateFormat.DEFAULT, properties.getDatePatternLocale());
    if (format instanceof SimpleDateFormat) {
      /*
       * Must use a localized pattern due to the DateFormatHelper's dependence
       * upon local pattern chars.
       */
      ((SimpleDateFormat) format).applyLocalizedPattern(properties
          .getDatePattern());
    } else {
      format = new SimpleDateFormat(properties.getDatePattern());
    }
    this.dateFormat = format;
  }

  /**
   * Returns the time part, used in a log filename suffix, for the period within
   * which the specified time falls.
   * <p>
   * Synchronized as used by both the appender and the log file scavenger.
   * 
   * @param timeForSuffix
   * @return Time part of a log filename.
   */
  final synchronized String toString(final long timeForSuffix) {
    this.calendar.setTimeInMillis(timeForSuffix);
    final Date suffixDate = this.calendar.getTime();
    return this.dateFormat.format(suffixDate);
  }
}
