/* license {{{
The contents of this file may be used under the terms of the GNU General Public
License Version 2 or later (the "GPL").

Copyright (C) 2003 Tomas Styblo <tripie@cpan.org>
}}} */

/* includes {{{ */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/XKBlib.h>
#include <X11/Xatom.h>
#include <gdk/gdkx.h>
#include <gkrellm2/gkrellm.h>
/* }}} */

/* include all the XPM files with flags {{{ */
#include "flags/ar.xpm"
#include "flags/be.xpm"
#include "flags/br.xpm"
#include "flags/cz.xpm"
#include "flags/cz_prog.xpm"
#include "flags/de.xpm"
#include "flags/dk.xpm"
#include "flags/ee.xpm"
#include "flags/el.xpm"
#include "flags/es.xpm"
#include "flags/fi.xpm"
#include "flags/fr.xpm"
#include "flags/gb.xpm"
#include "flags/hr.xpm"
#include "flags/hu.xpm"
#include "flags/it.xpm"
#include "flags/il.xpm"
#include "flags/lt.xpm"
#include "flags/lv.xpm"
#include "flags/nl.xpm"
#include "flags/no.xpm"
#include "flags/pl.xpm"
#include "flags/ru.xpm"
#include "flags/se.xpm"
#include "flags/sk.xpm"
#include "flags/us.xpm"
#include "flags/unknown.xpm"
#include "caps_on.xpm"
#include "caps_off.xpm"
#include "num_on.xpm"
#include "num_off.xpm"
/* }}} */

/* structs {{{ */
typedef struct {
    char *name;
    char **flag_xpm;
} xkbGroup;
/* }}} */

/* macros {{{ */
#define XKB_CTRLS_MASK (XkbAllControlsMask & ~(XkbInternalModsMask | XkbIgnoreLockModsMask))
#define XKB_MAX_GROUPS XkbNumKbdGroups
#define CONFIG_KEYWORD "xkb"
#define PLUGIN_PLACEMENT MON_UPTIME
#define PLUGIN_NAME "xkb"
#define FULL_UPDATES 1
#define PANEL_HEIGHT 18
#define STARTS_WITH(group, flag) !strncmp(group, flag, strlen(flag))
#define MAX_PROPERTY_VALUE_LEN 4096
/* }}} */

/* public funcs {{{ */
GkrellmMonitor *gkrellm_init_plugin(void);
/* }}} */

/* static funcs {{{ */ 
static void xkb_error(char *msg);
static Bool xkb_init(Display *disp);
static Bool xkb_status(Display *disp, 
        int *num_groups, int *active_group, xkbGroup *groups);
static Bool xkb_get_groups_num(Display *disp, XkbDescPtr kb, int *num_groups);
static Bool xkb_get_groups_names(Display *disp, XkbDescPtr kb, 
        int num_groups, xkbGroup *groups);
static void xkb_find_flag_from_name(xkbGroup *group);
static Bool xkb_find_flag_from_rules(gchar **xkb_rules,
        int group_num, xkbGroup *group);
static gchar **parse_xkb_rules_names();
static Bool xkb_get_active_group(Display *disp, int *active_group);
static void xkb_free_groups(int num_groups, xkbGroup *groups);
static void update_plugin();
static gint panel_expose_event(GtkWidget *widget, GdkEventExpose *ev, gpointer gw);
static void keyboard_selection(GtkMenuItem *menuitem, gpointer data);
static void button_press_event(GtkWidget *widget, GdkEventButton *ev);
static void create_plugin(GtkWidget *vbox, gint first_create);
static void draw_flag();
static void cycle_layout();
static gchar *get_property (Display *disp, Window win,
        Atom xa_prop_type, gchar *prop_name, unsigned long *size);
/* }}} */

/* static data {{{ */
static GkrellmPanel *panel;
static GkrellmMonitor *monitor;
static GkrellmTicks *pGK;
static int force_redraw = 0;
static int update_locked = 0;

static struct {
    const char *country_code;
    char **flag_pointer; 
} global_flag_map[] = {
    { "ar", flag_xpm_ar },
    { "be", flag_xpm_be },
    { "br", flag_xpm_br },
    { "cz", flag_xpm_cz },
    { "cz_prog", flag_xpm_cz_prog },
    { "de", flag_xpm_de },
    { "dk", flag_xpm_dk },
    { "ee", flag_xpm_ee },
    { "el", flag_xpm_el },
    { "es", flag_xpm_es },
    { "fi", flag_xpm_fi },
    { "fr", flag_xpm_fr },
    { "gb", flag_xpm_gb },
    { "hr", flag_xpm_hr },
    { "hu", flag_xpm_hu },
    { "it", flag_xpm_it },
    { "il", flag_xpm_il },
    { "lt", flag_xpm_lt },
    { "lv", flag_xpm_lv },
    { "nl", flag_xpm_nl },
    { "no", flag_xpm_no },
    { "pl", flag_xpm_pl },
    { "ru", flag_xpm_ru },
    { "se", flag_xpm_se },
    { "sk", flag_xpm_sk },
    { "us", flag_xpm_us },
    { NULL, NULL }
};

static struct {
    const char *country_name;
    char **flag_pointer; 
} global_names_map[] = {
    { "Arabic", flag_xpm_ar },
    { "Belgian", flag_xpm_be },
    { "Brazilian", flag_xpm_br },
    { "Czech Prog", flag_xpm_cz_prog },
    { "Czech", flag_xpm_cz },
    { "Croatian", flag_xpm_hr },
    { "Danish", flag_xpm_dk },
    { "Estonian", flag_xpm_ee },
    { "Finnish", flag_xpm_fi },
    { "French", flag_xpm_fr },
    { "German", flag_xpm_de },
    { "Great Britain", flag_xpm_gb },
    { "Hungarian", flag_xpm_hu },
    { "Italian", flag_xpm_it },
    { "Israelian", flag_xpm_il },
    { "ISO8859-7", flag_xpm_el }, /* greek */
    { "Latvian", flag_xpm_lv },
    { "Lithuanian", flag_xpm_lt },
    { "Nederland", flag_xpm_nl },
    { "Norwegian", flag_xpm_no },
    { "Polish", flag_xpm_pl },
    { "Russian", flag_xpm_ru },
    { "Spanish", flag_xpm_es },
    { "Swedish", flag_xpm_se },
    { "Slovak", flag_xpm_sk },
    { "US/ASCII", flag_xpm_us },
    { "en_US", flag_xpm_us },
    { NULL, NULL }
};

/* }}} */

static GkrellmMonitor plugin_mon = { /* {{{ */
    PLUGIN_NAME,         /* Name, for config tab.        */
    0,              /* Id,  0 if a plugin           */
    create_plugin,  /* The create_plugin() function */
    update_plugin,  /* The update_plugin() function */
    NULL,           /* The create_plugin_tab() config function */
    NULL,           /* The apply_plugin_config() function      */

    NULL,           /* The save_plugin_config() function  */
    NULL,           /* The load_plugin_config() function  */
    CONFIG_KEYWORD, /* config keyword                     */

    NULL,           /* Undefined 2  */
    NULL,           /* Undefined 1  */
    NULL,           /* Undefined 0  */

    PLUGIN_PLACEMENT, /* Insert plugin before this monitor.       */
    NULL,           /* Handle if a plugin, filled in by GKrellM */
    NULL            /* path if a plugin, filled in by GKrellM   */
};
/* }}} */

GkrellmMonitor *gkrellm_init_plugin(void) { /* {{{ */
    monitor = &plugin_mon;
    pGK = gkrellm_ticks();
    return monitor;
}
/* }}} */

/* main() function for testing {{{
int main(int argc, char **argv) {
    Display *disp;
    int num_groups;
    xkbGroup groups[XKB_MAX_GROUPS];
    int active_group;

    if (! (disp = XOpenDisplay(NULL))) {
        xkb_error("XOpenDisplay()");
        return EXIT_FAILURE;
    }

    if (! xkb_init(disp))
        return EXIT_FAILURE;

    while (1) { 
        if (! xkb_status(disp, &num_groups, &active_group, groups))
            return EXIT_FAILURE;

        printf("num_groups: %d\n", num_groups);
        printf("active_group: %d\n", active_group);
        printf("active group name: %s\n", groups[active_group].name);
        printf("active group flag: %s\n", groups[active_group].flag_xpm[0]);
        sleep(2);
    }
   
    xkb_free_groups(groups);
    XCloseDisplay(disp);
    
    return EXIT_SUCCESS;
} 
}}} */

static void update_plugin() { /* {{{ */
    if (update_locked) return;
    update_locked = 1;
    if (FULL_UPDATES || pGK->second_tick) {
        draw_flag(0);
    }
    update_locked = 0;
}
/* }}} */

static gint panel_expose_event(GtkWidget *widget, /* {{{ */
        GdkEventExpose *ev, gpointer gw) { 
    force_redraw = 1;
    update_plugin();
    return FALSE;
}
/* }}} */

static void button_press_event(GtkWidget *widget, /* {{{ */
        GdkEventButton *ev) {
    GtkWidget *menu = NULL;
    GtkWidget *item = NULL;
    int num_groups = 0;
    xkbGroup groups[XKB_MAX_GROUPS];
    int active_group;
    int i;
        
    if (! widget) return;

    if (ev->button == 1) {
        memset(groups, 0, sizeof(groups));

        if (! xkb_status(GDK_DISPLAY(), &num_groups, &active_group, groups))
            goto cleanup;

        menu = gtk_menu_new();

        /* Create the menu items */
        for (i = 0; i < num_groups; i++) {
            GdkPixmap *img_gdk;
            GtkWidget *img;
            
            if (! (img_gdk = gdk_pixmap_colormap_create_from_xpm_d(NULL,
                    gdk_colormap_get_system (), NULL, NULL, groups[i].flag_xpm)))
                continue;
            if (! (img = gtk_image_new_from_pixmap(img_gdk, NULL)))
                continue;
            item = gtk_image_menu_item_new_with_label(groups[i].name ? groups[i].name : "N/A");
            gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
            gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(item), img);
            g_signal_connect(G_OBJECT(item), "activate",
                    G_CALLBACK(keyboard_selection), GINT_TO_POINTER(i));
            gtk_widget_show(item);
            gtk_widget_show(img);
        }

        gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, ev->button, ev->time);

        cleanup:
        xkb_free_groups(num_groups, groups);
    } else if (ev->button == 3) {
        cycle_layout();
    }
}
/* }}} */

static void keyboard_selection(GtkMenuItem *menuitem, /* {{{ */
        gpointer data) {
    /* fprintf(stderr, "selected: %d\n", GPOINTER_TO_INT(data)); */
    XkbLockGroup(GDK_DISPLAY(), XkbUseCoreKbd, GPOINTER_TO_INT(data));
    XSync(GDK_DISPLAY(), False);
    update_plugin();
}
/* }}} */

static void create_plugin(GtkWidget *vbox, gint first_create) { /* {{{ */
    if (first_create) {
        panel = gkrellm_panel_new0();
        xkb_init(GDK_DISPLAY());
    }
  	gkrellm_panel_configure(panel, NULL, NULL);
  	gkrellm_panel_configure_set_height(panel, PANEL_HEIGHT);
	gkrellm_panel_create(vbox, monitor, panel);
    if (first_create) {
        g_signal_connect(G_OBJECT(panel->drawing_area),
            "expose_event", G_CALLBACK(panel_expose_event), NULL);
        g_signal_connect(G_OBJECT(panel->drawing_area), 
            "button_press_event", G_CALLBACK(button_press_event), NULL);
    }
    draw_flag(1);
}
/* }}} */

static void draw_flag(int redraw) { /* {{{ */
    static char **current_flag = NULL;
    int num_groups = 0;
    xkbGroup groups[XKB_MAX_GROUPS];
    int active_group;
    int flag_width, flag_height;
    int loc_x, loc_y;
    static GdkPixmap *flag_pix = NULL;
    static GdkPixmap *caps_pix = NULL;
    static GdkPixmap *num_pix = NULL;
    static GdkPixmap *bg_scaled = NULL;
    static GkrellmDecal *decal_flag = NULL;
    static GkrellmDecal *decal_caps = NULL;
    static GkrellmDecal *decal_num = NULL;
    static GkrellmDecal *decal_bg = NULL;
    GkrellmPiximage *bg = NULL;
    static GtkTooltips *tooltip = NULL;
    unsigned int ind_status;
    static int ind_caps_prev, ind_num_prev;
    int ind_caps, ind_num;

    memset(groups, 0, sizeof(groups));
    
    if (! xkb_status(GDK_DISPLAY(), &num_groups, &active_group, groups))
        goto cleanup;
  
    if (active_group < 0)
        goto cleanup;
    
    if (XkbGetIndicatorState(GDK_DISPLAY(), XkbUseCoreKbd, &ind_status) == Success) {
        ind_caps = ind_status & 1<<0;
        ind_num = ind_status & 1<<1;
        /*
        fprintf(stderr, "caps: %d num: %d scroll: %d\n", 
                ind_status & 1<<0, ind_status & 1<<1, ind_status & 1<<2);
        */
    }
    else {
        goto cleanup;
    }
    
    if (! (redraw || force_redraw ||
           current_flag == NULL || current_flag != groups[active_group].flag_xpm ||
           ind_caps != ind_caps_prev || ind_num != ind_num_prev))
        goto cleanup;

    /* fprintf(stderr, "Redrawing...\n"); */
    
    ind_caps_prev = ind_caps;
    ind_num_prev = ind_num;
        
    current_flag = groups[active_group].flag_xpm;

    if (!redraw && decal_bg) {
        gkrellm_destroy_decal(decal_bg); decal_bg = NULL;
    }
    if (!redraw && decal_flag) {
        gkrellm_destroy_decal(decal_flag); decal_flag = NULL;
    }
    if (!redraw && decal_caps) {
        gkrellm_destroy_decal(decal_caps); decal_caps = NULL;
    }
    if (!redraw && decal_num) {
        gkrellm_destroy_decal(decal_num); decal_num = NULL;
    }
    if (flag_pix) {
        gkrellm_free_pixmap(&flag_pix); flag_pix = NULL;
    }
    if (flag_pix) {
        gkrellm_free_pixmap(&flag_pix); flag_pix = NULL;
    }
    if (caps_pix) {
        gkrellm_free_pixmap(&caps_pix); caps_pix = NULL;
    }
    if (num_pix) {
        gkrellm_free_pixmap(&num_pix); num_pix = NULL;
    }
    if (bg_scaled) {
        gkrellm_free_pixmap(&bg_scaled); bg_scaled = NULL;
    }

    if ((bg = gkrellm_bg_meter_piximage(DEFAULT_STYLE_ID))) {
        gkrellm_scale_piximage_to_pixmap(bg, &bg_scaled, NULL, 
                gkrellm_chart_width(), PANEL_HEIGHT);
    }
    
    if ((flag_pix = gdk_pixmap_colormap_create_from_xpm_d(NULL, 
                    gtk_widget_get_default_colormap(), 
                    NULL, NULL, current_flag)) && 
        (caps_pix = gdk_pixmap_colormap_create_from_xpm_d(NULL, 
                    gtk_widget_get_default_colormap(), 
                    NULL, NULL, ind_caps ? caps_on_xpm : caps_off_xpm)) && 
        (num_pix = gdk_pixmap_colormap_create_from_xpm_d(NULL, 
                    gtk_widget_get_default_colormap(), 
                    NULL, NULL, ind_num ? num_on_xpm : num_off_xpm))) {
        gdk_drawable_get_size(flag_pix, &flag_width, &flag_height);
        loc_x = (gkrellm_chart_width() - flag_width) / 2;
        loc_y = 2;
        if (bg_scaled) {
            decal_bg = gkrellm_create_decal_pixmap(panel, bg_scaled, NULL, 0, NULL, 0, 0);
            gkrellm_draw_decal_pixmap(panel, decal_bg, 0);
        }
        decal_flag = gkrellm_create_decal_pixmap(panel, flag_pix, NULL, 0, NULL, loc_x, loc_y);
        decal_caps = gkrellm_create_decal_pixmap(panel, caps_pix, NULL, 0, NULL, 0, loc_y);
        decal_num = gkrellm_create_decal_pixmap(panel, num_pix, NULL, 0, NULL, 
                gkrellm_chart_width() - 14, loc_y);
        gkrellm_draw_decal_pixmap(panel, decal_flag, 0);
        gkrellm_draw_decal_pixmap(panel, decal_caps, 0);
        gkrellm_draw_decal_pixmap(panel, decal_num, 0);
        gkrellm_draw_panel_layers(panel);
        /*
        printf("active group name: %s\n", groups[active_group].name);
        */
    }
  
    if (! tooltip) tooltip = gtk_tooltips_new();
    if (groups[active_group].name) {
        gtk_tooltips_set_tip(tooltip, panel->drawing_area,
            groups[active_group].name , NULL);
    }
    
    if (redraw)
        current_flag = NULL;

    if (force_redraw)
        force_redraw = 0;
    
    cleanup:
        xkb_free_groups(num_groups, groups);
}
/* }}} */

static Bool xkb_init(Display *disp) { /* {{{ */
    if (XkbQueryExtension(disp, NULL, NULL, NULL, NULL, NULL)) {
        return True;
    }
    else {
        xkb_error("Cannot initialize XKB extension.");
        return False;
    }
}
/* }}} */

static Bool xkb_status(Display *disp, int *num_groups, /* {{{ */
                       int *active_group, xkbGroup *groups) {
    Bool status = False;
    int i;
    XkbDescPtr kb;
    gchar **xkb_rules = NULL;
    
    if (! (kb = XkbAllocKeyboard())) {
        xkb_error("XkbAllocKeyboard()");
        goto cleanup;
    }

    if (! xkb_get_groups_num(disp, kb, num_groups))
        goto cleanup;
    
    if (! xkb_get_groups_names(disp, kb, *num_groups, groups))
        goto cleanup;
    
    if (! xkb_get_active_group(disp, active_group))
        goto cleanup;
  
    xkb_rules = parse_xkb_rules_names();
    
    for (i = 0; i < *num_groups; i++) {
        if (! (xkb_rules && xkb_find_flag_from_rules(xkb_rules, i, &groups[i]))) {
            xkb_find_flag_from_name(&groups[i]);
        }
    }

    if (xkb_rules) g_strfreev(xkb_rules);
    
    status = True;
    
    cleanup:
        if (kb) XkbFreeKeyboard(kb, XkbAllComponentsMask, True);
        return status;
}
/* }}} */

static Bool xkb_get_groups_num(Display *disp, XkbDescPtr kb, /* {{{ */
                               int *num_groups) {
    if (XkbGetControls(disp, XKB_CTRLS_MASK, kb) == Success) {
        *num_groups = kb->ctrls->num_groups;
        XkbFreeControls(kb, XKB_CTRLS_MASK, True);
        return True;
    }
    else {
        xkb_error("XkbGetControls()");
        return False;
    }
}
/* }}} */

static Bool xkb_get_groups_names(Display *disp, XkbDescPtr kb, /* {{{ */
                                 int num_groups, xkbGroup *groups) {
    Bool status = False;
    char *name = NULL;
    int i;

    if (XkbGetNames(disp, XkbGroupNamesMask, kb) != Success) {
        xkb_error("XkbGetNames()");
        return False;
    }
  
    for (i = 0; i < num_groups; i++) {
        if (kb->names->groups[i]) {
            if ((name = XGetAtomName(disp, kb->names->groups[i]))) {
                groups[i].name = name;
            }
            else {
                xkb_error("XGetAtomName()");
                groups[i].name = NULL; 
            }
        }
        else {
            /*
            fprintf(stderr, "gkrellm-xkb: Keyboard group description is incomplete.\n"
                            "*** gkrellm-xkb does NOT work in GNOME or KDE. ***\n");
            */
            groups[i].name = NULL; 
        }
    }

    status = True;
    
    XkbFreeNames(kb, XkbGroupNamesMask, True);
    return status;
}
/* }}} */

static Bool xkb_get_active_group(Display *disp, int *active_group) { /* {{{ */
    XkbStateRec state;

    if (XkbGetState(disp, XkbUseCoreKbd, &state) == Success) {
        *active_group = state.group;        
        return True;
    }
    else {
        xkb_error("XkbGetState()");
        return False;
    }
}
/* }}} */

static Bool xkb_find_flag_from_rules(gchar **xkb_rules, /* {{{ */
        int group_num, xkbGroup *group) {
    gint pos = 0;
    
    while (xkb_rules[pos]) {
        if (pos == group_num) {
            int i;
            i = 0;
            while (global_flag_map[i].country_code) {
                if (g_ascii_strcasecmp(global_flag_map[i].country_code, xkb_rules[pos]) == 0) {
                    group->flag_xpm = global_flag_map[i].flag_pointer;
                    /* fprintf(stderr, "xkb_find_flag_from_rules: found flag: %s\n", xkb_rules[pos]); */
                    return True;
                }
                i++;
            }
        }
        pos++;
    }

    return False;
}
/* }}} */

static gchar **parse_xkb_rules_names() { /* {{{ */
    gchar *prop_rules = NULL;
    gchar *country_codes = NULL;
    unsigned long prop_rules_len;
    int i, pos;
    gchar **country_codes_array = NULL;

    /* get the _XKB_RULES_NAMES from the root window */
    if (! (prop_rules = get_property(GDK_DISPLAY(), 
                    DefaultRootWindow(GDK_DISPLAY()), XA_STRING,
                    "_XKB_RULES_NAMES", &prop_rules_len))) {
        goto cleanup;
    }
    
    /* find the country codes separated with a comma */
    pos = 1;
    for (i = 0; i < prop_rules_len; i++) {
        if (prop_rules[i] == '\0' && ++pos == 3) {
            country_codes = prop_rules + i + 1;
            break;
        }
    }
    
    if (! country_codes) goto cleanup;
    /* fprintf(stderr, "country_codes: %s\n", country_codes); */
   
    /* parse country_codes string into array */
    country_codes_array = g_strsplit(country_codes, ",", 0);
    
    cleanup:
        if (prop_rules) g_free(prop_rules);
        return country_codes_array;
}
/* }}} */

static void xkb_find_flag_from_name(xkbGroup *group) { /* {{{ */
    int i;
    
    group->flag_xpm = NULL;
   
    /* fprintf(stderr, "xkb_find_flag_from_name\n"); */
    
    if (! group->name) {
        group->flag_xpm = flag_xpm_unknown;
        return;
    }

    i = 0;
    while (global_names_map[i].country_name) {
        if (STARTS_WITH(group->name, global_names_map[i].country_name)) {
            group->flag_xpm = global_names_map[i].flag_pointer;
            return;
        }
        i++;
    }
    
    if (group->flag_xpm == NULL) group->flag_xpm = flag_xpm_unknown;
    
    return;
}
/* }}} */

static void xkb_error(char *msg) { /* {{{ */
    fprintf(stderr, "gkrellm-xkb error: %s\n", msg);
}
/* }}} */

static void xkb_free_groups(int num_groups, xkbGroup *groups) { /* {{{ */
    int i;

    for (i = 0; i < num_groups; i++) {
        if (groups[i].name) XFree(groups[i].name);
    }
}
/* }}} */

static void cycle_layout() { /* {{{ */
    int active_group, next_group, num_groups = 0;
    XkbDescPtr kb;
    
    if (! (kb = XkbAllocKeyboard())) {
        xkb_error("XkbAllocKeyboard()");
        goto cleanup;
    }
    
    if (!xkb_get_groups_num(GDK_DISPLAY(), kb, &num_groups))
        goto cleanup;
    
    if (!xkb_get_active_group(GDK_DISPLAY(), &active_group))
        goto cleanup;

    next_group = active_group + 1;
    if (next_group == num_groups)
        next_group = 0;

    keyboard_selection(NULL/*menuitem*/, GINT_TO_POINTER(next_group));

    cleanup:
        if (kb) XkbFreeKeyboard(kb, XkbAllComponentsMask, True);
}
/* }}} */

static gchar *get_property (Display *disp, Window win, /*{{{*/
        Atom xa_prop_type, gchar *prop_name, unsigned long *size) {
    Atom xa_prop_name;
    Atom xa_ret_type;
    int ret_format;
    unsigned long ret_nitems;
    unsigned long ret_bytes_after;
    unsigned long tmp_size;
    unsigned char *ret_prop;
    gchar *ret;
    
    xa_prop_name = XInternAtom(disp, prop_name, False);
    
    if (XGetWindowProperty(disp, win, xa_prop_name, 0, MAX_PROPERTY_VALUE_LEN / 4, False,
            xa_prop_type, &xa_ret_type, &ret_format,     
            &ret_nitems, &ret_bytes_after, &ret_prop) != Success) {
        /* fprintf(stderr, "Cannot get %s property.\n", prop_name); */
        return NULL;
    }
  
    if (xa_ret_type != xa_prop_type) {
        /* fprintf(stderr, "Invalid type of %s property.\n", prop_name); */
        XFree(ret_prop);
        return NULL;
    }

    /* null terminate the result to make string handling easier */
    tmp_size = (ret_format / 8) * ret_nitems;
    ret = g_malloc(tmp_size + 1);
    memcpy(ret, ret_prop, tmp_size);
    ret[tmp_size] = '\0';

    if (size) {
        *size = tmp_size;
    }
    
    XFree(ret_prop);
    return ret;
}/*}}}*/

