/*
 * uterm - Linux User-Space Terminal
 *
 * Copyright (c) 2011-2013 David Herrmann <dh.herrmann@googlemail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/* Internal definitions */

#ifndef UTERM_VIDEO_INTERNAL_H
#define UTERM_VIDEO_INTERNAL_H

#include <inttypes.h>
#include <limits.h>
#include <stdbool.h>
#include <stdlib.h>
#include "eloop.h"
#include "shl_dlist.h"
#include "shl_hook.h"
#include "uterm_video.h"

/* backend-operations */

struct display_ops {
	int (*init)(struct uterm_display *display);
	void (*destroy)(struct uterm_display *display);
	int (*set_dpms)(struct uterm_display *disp, int state);
	int (*use)(struct uterm_display *disp);
	int (*swap)(struct uterm_display *disp);
	bool (*is_swapping)(struct uterm_display *disp);
	bool (*need_redraw)(struct uterm_display *disp);
	int (*fake_blendv)(struct uterm_display *disp, const struct uterm_video_blend_req *req,
			   size_t num);
	int (*fill)(struct uterm_display *disp, uint8_t r, uint8_t g, uint8_t b, unsigned int x,
		    unsigned int y, unsigned int width, unsigned int height);
	void (*set_damage)(struct uterm_display *disp, size_t n_rect,
			   struct uterm_video_rect *damages);
	bool (*has_damage)(struct uterm_display *disp);
};

struct video_ops {
	int (*init)(struct uterm_video *video, const char *node);
	void (*destroy)(struct uterm_video *video);
	int (*poll)(struct uterm_video *video);
	void (*sleep)(struct uterm_video *video);
	int (*wake_up)(struct uterm_video *video);
};

struct uterm_video_module {
	const char *name;
	struct shl_module *owner;
	const struct video_ops ops;
};

#define VIDEO_CALL(func, els, ...) (func ? func(__VA_ARGS__) : els)

/* uterm_display */

#define DISPLAY_ONLINE 0x01
#define DISPLAY_VSYNC 0x02
#define DISPLAY_AVAILABLE 0x04
#define DISPLAY_OPEN 0x08
#define DISPLAY_DBUF 0x10
#define DISPLAY_DITHERING 0x20
#define DISPLAY_PFLIP 0x40
#define DISPLAY_OPENGL 0x80
#define DISPLAY_INUSE 0x100
#define DISPLAY_DAMAGE 0x200

struct uterm_display {
	char *name;
	struct shl_dlist list;
	unsigned long ref;
	unsigned int flags;
	unsigned int width;
	unsigned int height;

	struct uterm_video *video;

	struct shl_hook *hook;
	int dpms;

	const struct display_ops *ops;
	void *data;
};

int display_new(struct uterm_display **out, const struct display_ops *ops,
		struct uterm_video *video, const char *name);
int uterm_display_bind(struct uterm_display *disp);
void uterm_display_unbind(struct uterm_display *disp);
void uterm_display_ready(struct uterm_display *disp);

#define DISPLAY_CB(disp, act)                                                                      \
	shl_hook_call((disp)->hook, (disp),                                                        \
		      &(struct uterm_display_event){                                               \
			      .action = (act),                                                     \
		      })

static inline bool display_is_online(const struct uterm_display *disp)
{
	return disp->video && (disp->flags & DISPLAY_ONLINE);
}

/* uterm_video */

#define VIDEO_AWAKE 0x01
#define VIDEO_HOTPLUG 0x02

struct uterm_video {
	unsigned long ref;
	unsigned int flags;
	struct ev_eloop *eloop;

	struct shl_dlist displays;
	struct shl_hook *hook;

	bool use_original;
	unsigned int desired_width;
	unsigned int desired_height;

	const struct uterm_video_module *mod;
	void *data;
};

static inline bool video_is_awake(const struct uterm_video *video)
{
	return video->flags & VIDEO_AWAKE;
}

static inline bool video_need_hotplug(const struct uterm_video *video)
{
	return video->flags & VIDEO_HOTPLUG;
}

#define VIDEO_CB(vid, disp, act)                                                                   \
	shl_hook_call((vid)->hook, (vid),                                                          \
		      &(struct uterm_video_hotplug){                                               \
			      .display = (disp),                                                   \
			      .action = (act),                                                     \
		      })
#endif /* UTERM_VIDEO_INTERNAL_H */
