/* NVTV NVidia CRTC data -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: data_nv.c,v 1.23 2004/01/30 08:47:02 dthierbach Exp $
 *
 * Contents:
 *
 * Data routines for the NVidia CRTC data.
 *
 */

#include "local.h" /* before everything else */

#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include "data_bt.h"
#include "data_cx.h"
#include "data_ch.h"
#include "data_ph.h"
#include "data_nx.h"
#include "data_nv.h"
#include "calc_bt.h"

/* ======== BT ================================ */

#define TV_DEF_BT (TV_CAP_MACROVISION | TV_CAP_NONINTERLACED | \
		   TV_CAP_MONOCHROME)

#define TV_DEF_BT_DUAL (TV_DEF_BT | TV_DEF_DUALVIEW)

/* -------- BT -------- NTSC -------- */

/* -------- 640x480 -------- */

TVNvRegs nv_bt_ntsc_small_a = { 
  HDisplay    : 640,
  HSyncStart  : 664-8,
  HSyncEnd    : 696-8,
  HTotal      : 744,
  VDisplay    : 480,
  VSyncStart  : 524,
  VSyncEnd    : 526,
  VTotal      : 600,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_ntsc_normal_a = { 
  HDisplay    : 640,
  HSyncStart  : 680-8,
  HSyncEnd    : 736-8,
  HTotal      : 752,
  VDisplay    : 480,
  VSyncStart  : 506,
  VSyncEnd    : 508,
  VTotal      : 562,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_ntsc_huge_a = { 
  HDisplay    : 640,
  HSyncStart  : 672-8,
  HSyncEnd    : 696-8,
  HTotal      : 760,
  VDisplay    : 480,
  VSyncStart  : 486,
  VSyncEnd    : 488,
  VTotal      : 521,
  flags       : 0,
  clock       : 0,
};

/* -------- 800x600 -------- */

TVNvRegs nv_bt_ntsc_small_b = { 
  HDisplay    : 800,
  HSyncStart  : 808-8,
  HSyncEnd    : 832-8,
  HTotal      : 848,
  VDisplay    : 600,
  VSyncStart  : 630,
  VSyncEnd    : 632,
  VTotal      : 718,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_ntsc_normal_b = { /* FIXME: SyncStart ok ?? */
  HDisplay    : 800,
  HSyncStart  : 800-8,
  HSyncEnd    : 824-8,
  HTotal      : 840,
  VDisplay    : 600,
  VSyncStart  : 630,
  VSyncEnd    : 632,
  VTotal      : 690,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_ntsc_huge_b = { 
  HDisplay    : 800,
  HSyncStart  : 808-8,
  HSyncEnd    : 824-8,
  HTotal      : 848,
  VDisplay    : 600,
  VSyncStart  : 608,
  VSyncEnd    : 610,
  VTotal      : 650,
  flags       : 0,
  clock       : 0,
};

/* -------- 720x480 -------- */

TVNvRegs nv_bt_ntsc_normal_c = { 
  HDisplay    : 720,
  HSyncStart  : 796-8,
  HSyncEnd    : 936-8,
  HTotal      : 896,
  VDisplay    : 480,
  VSyncStart  : 527,
  VSyncEnd    : 557,
  VTotal      : 600,
  flags       : 0,
  clock       : 0,
};

/* -------- 720x480 DVD -------- */

TVNvRegs nv_bt_ntsc_dvd_a = { 
  HDisplay    : 720,
  HSyncStart  : 744,
  HSyncEnd    : 808,
  HTotal      : 832,
  VDisplay    : 480,
  VSyncStart  : 489,
  VSyncEnd    : 491,
  VTotal      : 525,
  flags       : 0,
  clock       : 0,
};

/* The following mode flickers at the top part */

TVNvRegs nv_bt_ntsc_dvd_b = { /* TV_DEF_BT_DUAL */
  HDisplay    : 720,
  HSyncStart  : 760, /* Beos: 744 */
  HSyncEnd    : 840, /* Beos: 744+144 */
  HTotal      : 848,
  VDisplay    : 480,
  VSyncStart  : 490,
  VSyncEnd    : 492, /* Beos: 490+26 */
  VTotal      : 525,
  flags       : 0,
  clock       : 0,
};

/* -------- BT -------- PAL -------- */

/* -------- 640x480 -------- */

TVNvRegs nv_bt_pal_small_a = { 
  HDisplay    : 640,
  HSyncStart  : 744-8,
  HSyncEnd    : 888-8,
  HTotal      : 944,
  VDisplay    : 480,
  VSyncStart  : 532,
  VSyncEnd    : 534,
  VTotal      : 625,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_pal_normal_a = { /* no mon sync !! */
  HDisplay    : 640,
  HSyncStart  : 728, /* 768-8, */
  HSyncEnd    : 752, /* 880-8, */
  HTotal      : 1000,
  VDisplay    : 480,
  VSyncStart  : 508,
  VSyncEnd    : 510,
  VTotal      : 565,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_pal_huge_a = { /* no mon sync !! */
  HDisplay    : 640,
  HSyncStart  : 736-8,
  HSyncEnd    : 872-8,
  HTotal      : 976,
  VDisplay    : 480,
  VSyncStart  : 483,
  VSyncEnd    : 485,
  VTotal      : 520,
  flags       : 0,
  clock       : 0,
};

/* -------- 800x600 -------- */

TVNvRegs nv_bt_pal_small_b = {
  HDisplay    : 800,
  HSyncStart  : 824-8, /* 880, */
  HSyncEnd    : 880-8, /* 912, */
  HTotal      : 912,
  VDisplay    : 600,
  VSyncStart  : 658,
  VSyncEnd    : 660,
  VTotal      : 750,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_pal_normal_b = {
  HDisplay    : 800,
  HSyncStart  : 840-8,
  HSyncEnd    : 864-8,
  HTotal      : 952,
  VDisplay    : 600,
  VSyncStart  : 625,
  VSyncEnd    : 627,
  VTotal      : 687,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_pal_large_b = {
  HDisplay    : 800,
  HSyncStart  : 848-8,
  HSyncEnd    : 928-8,
  HTotal      : 1000,
  VDisplay    : 600,
  VSyncStart  : 615,
  VSyncEnd    : 617,
  VTotal      : 665,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_pal_huge_b = {
  HDisplay    : 800,
  HSyncStart  : 944-8,
  HSyncEnd    : 968-8,
  HTotal      : 1000,
  VDisplay    : 600,
  VSyncStart  : 653,
  VSyncEnd    : 657,
  VTotal      : 696,
  flags       : 0,
  clock       : 0,
};

/* -------- 768x576 -------- */

TVNvRegs nv_bt_pal_small_c = { 
  HDisplay    : 768,
  HSyncStart  : 824-8,
  HSyncEnd    : 896-8,
  HTotal      : 976,
  VDisplay    : 576,
  VSyncStart  : 610,
  VSyncEnd    : 621,
  VTotal      : 680,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_pal_normal_c = { 
  HDisplay    : 768,
  HSyncStart  : 824-8,
  HSyncEnd    : 896-8,
  HTotal      : 968,
  VDisplay    : 576,
  VSyncStart  : 599,
  VSyncEnd    : 601,
  VTotal      : 655,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_pal_huge_c = { 
  HDisplay    : 768,
  HSyncStart  : 832,
  HSyncEnd    : 856,
  HTotal      : 944,
  VDisplay    : 576,
  VSyncStart  : 583,
  VSyncEnd    : 585,
  VTotal      : 625,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_pal_normal1_c = { /* TV_DEF_BT_DUAL */
  HDisplay    : 768,
  HSyncStart  : 824-8,
  HSyncEnd    : 896-8,
  HTotal      : 968,
  VDisplay    : 576,
  VSyncStart  : 600,
  VSyncEnd    : 602,
  VTotal      : 665,
  flags       : 0,
  clock       : 0,
};

/* -------- 800x576 -------- */

/* 800 x 576 (for 1024x576 widescreen scaled by xine) */

TVNvRegs nv_bt_pal_normal_d = { 
  HDisplay    : 800,
  HSyncStart  : 840-8,
  HSyncEnd    : 888-8,
  HTotal      : 960,
  VDisplay    : 576,
  VSyncStart  : 608,
  VSyncEnd    : 610,
  VTotal      : 675,
  flags       : 0,
  clock       : 0,
};

/* -------- 720x576 DVD -------- */

TVNvRegs nv_bt_pal_dvd_a = { 
  HDisplay    : 720,
  HSyncStart  : 746, /* Beos: 744 */
  HSyncEnd    : 832, /* Beos: 744+140 -- not a multiple of 8 */
  HTotal      : 840, /* 888 */
  VDisplay    : 576,
  VSyncStart  : 579, /* Beos: 579 */
  VSyncEnd    : 581, /* Beos: 579+42=621 */
  VTotal      : 625,
  flags       : 0,
  clock       : 0,
};

/* -------- BT -------- PAL-60 -------- */

/* -------- 640x480 -------- */

TVNvRegs nv_bt_pal60_small_a = { 
  HDisplay    : 640,
  HSyncStart  : 664,
  HSyncEnd    : 728,
  HTotal      : 768,
  VDisplay    : 480,
  VSyncStart  : 532,
  VSyncEnd    : 534,
  VTotal      : 609,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_pal60_normal_a = { 
  HDisplay    : 640,
  HSyncStart  : 672,
  HSyncEnd    : 728,
  HTotal      : 784,
  VDisplay    : 480,
  VSyncStart  : 509,
  VSyncEnd    : 511,
  VTotal      : 567,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_pal60_huge_a = { 
  HDisplay    : 640,
  HSyncStart  : 656,
  HSyncEnd    : 696,
  HTotal      : 752,
  VDisplay    : 480,
  VSyncStart  : 489,
  VSyncEnd    : 491,
  VTotal      : 525,
  flags       : 0,
  clock       : 0,
};

/* -------- 800x600 -------- */

TVNvRegs nv_bt_pal60_small_b = { 
  HDisplay    : 800,
  HSyncStart  : 800,
  HSyncEnd    : 824,
  HTotal      : 832,
  VDisplay    : 600,
  VSyncStart  : 649,
  VSyncEnd    : 651,
  VTotal      : 735,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_pal60_normal_b = { 
  HDisplay    : 800,
  HSyncStart  : 800,
  HSyncEnd    : 824,
  HTotal      : 824,
  VDisplay    : 600,
  VSyncStart  : 625,
  VSyncEnd    : 627,
  VTotal      : 685,
  flags       : 0,
  clock       : 0,
};

TVNvRegs nv_bt_pal60_huge_b = { 
  HDisplay    : 800,
  HSyncStart  : 800,
  HSyncEnd    : 808,
  HTotal      : 824,
  VDisplay    : 600,
  VSyncStart  : 611,
  VSyncEnd    : 613,
  VTotal      : 655,
  flags       : 0,
  clock       : 0,
};

/* -------- 720x480 DVD -------- */

TVNvRegs nv_bt_pal60_dvd_a = { 
  HDisplay    : 720,
  HSyncStart  : 744,
  HSyncEnd    : 792,
  HTotal      : 856,
  VDisplay    : 480,
  VSyncStart  : 489,
  VSyncEnd    : 491,
  VTotal      : 525,
  flags       : 0,
  clock       : 0,
};

/* -------- BT async -------- PAL -------- */

/* -------- 640x480 -------- */

TVNvRegs nv_bt_async_pal_small_a = { 
  clock       : 29500, /* kHz */
  HDisplay    : 640,
  HSyncStart  : 744-8,
  HSyncEnd    : 888-8,
  HTotal      : 944,
  VDisplay    : 480,
  VSyncStart  : 532,
  VSyncEnd    : 534,
  VTotal      : 625,
  flags       : 0,
};

TVNvRegs nv_bt_slave_pal_small_a = { 
  clock       : 0, /* 29.500 kHz */
  HDisplay    : 640,
  HSyncStart  : 744-8,
  HSyncEnd    : 888-8,
  HTotal      : 944,
  VDisplay    : 480,
  VSyncStart  : 532,
  VSyncEnd    : 534,
  VTotal      : 625,
  flags       : 0,
  slave : {
    HTotal     : 944,
    HSyncStart : 1,
    HSyncEnd   : 64,
    VTotal     : 625,
    VSyncStart : 1,
    VSyncEnd   : 16,
    Unknown    : 0, /* anything bigger than 0 flickers ... */
  }
};

/* -------- BT overlay -------- PAL -------- */

/* -------- 640x480 -------- */

TVNvRegs nv_bt_ovly_pal_small_a = { 
  HDisplay    : 640,
  HSyncStart  : 768,
  HSyncEnd    : 80,
  VDisplay    : 480,
  VSyncStart  : 688,
  VSyncEnd    : 80,
  flags       : 0,
};

/* -------- BT modes -------- */

/*

Modes:

NTSC 640x480 S tv ok,  mon ok, oc ok, calcable (std 0)
NTSC 640x480 N tv ok,  mon ok, oc ok, calcable
NTSC 640x480 H tv ok,  mon ok, oc ok, calcable

NTSC 800x600 S tv ok,  mon ok, oc ok, calcable (std 2)
NTSC 800x600 N tv ok,  mon ok, oc ok, calcable 
NTSC 800x600 H tv ok*, mon ok, oc ok, calcable

PAL  640x480 S tv ok, mon ok, oc ok, calcable (std 1)
PAL  640x480 N tv ok, mon NO, oc ok, calcable
PAL  640x480 H tv  ?, mon  ?, oc ok, calcable

PAL  800x600 S tv ok, mon ok, oc ok, calcable (std 3)
PAL  800x600 N tv ok, mon ok, oc ok, calcable
PAL  800x600 H tv ok, mon NO, oc ?, not calcable

PAL  768x576 S tv ok, mon ok, oc ok, calcable
PAL  768x576 N tv ok, mon ok, oc ok, calcable

* needs several attempts, monitor takes some time syncing.

(Huge modes check with HWcursor = False)

Bad modes:

PAL  800x600 H distorted dualview
PAL  640x480 N no dualview
PAL  640x480 H no dualview
NTSC 800x600 N tv left border wrong (sync?)

*/

typedef struct {
  TVModeSpec spec;
  TVNvRegs *nv;
  TVBtRegs *bt;
  int hsynoffset;
  int hostFlags;
  int encFlags;
  int descFlags;
} TVTemplateNvBt;

TVTemplateNvBt templ_nv_bt [] = {
  {{TV_SYSTEM_NTSC, 640, 480, "Small",  "4:3", 13.79, 13.58},
   &nv_bt_ntsc_small_a,  &bt_ntsc_small_a,    0, 
   PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_NTSC, 640, 480, "Normal", "4:3", 10.59, 08.23},
   &nv_bt_ntsc_normal_a, &bt_ntsc_normal_a,  12, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_NTSC, 640, 480, "Huge",   "4:3", 02.46, 01.23}, 
   &nv_bt_ntsc_huge_a,   &bt_ntsc_huge_a,     0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},

  {{TV_SYSTEM_NTSC, 800, 600, "Small",  "4:3", 21.62, 11.52}, 
   &nv_bt_ntsc_small_b,  &bt_ntsc_small_b,   -8, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_NTSC, 800, 600, "Normal", "4:3", 11.90, 05.35}, 
   &nv_bt_ntsc_normal_b, &bt_ntsc_normal_b, -62, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_NTSC, 800, 600, "Huge",   "4:3", 07.15, 0.004},
   &nv_bt_ntsc_huge_b,   &bt_ntsc_huge_b,   -60, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},

  {{TV_SYSTEM_NTSC, 720, 480, "Normal", "DVD", 08.762, 18.107},
   &nv_bt_ntsc_normal_c, &bt_ntsc_normal_c,  -4, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_NTSC, 720, 480, "DVD",    "DVD", 01.245, 01.235},
   &nv_bt_ntsc_dvd_a,    &bt_ntsc_dvd_a,      0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},

  {{TV_SYSTEM_PAL, 640, 480, "Small",  "4:3", 16.56, 16.67},
   &nv_bt_pal_small_a,  &bt_pal_small_a,    0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_PAL, 640, 480, "Normal", "4:3", 12.87, 07.64}, 
   &nv_bt_pal_normal_a, &bt_pal_normal_a,   0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT},
  {{TV_SYSTEM_PAL, 640, 480, "Huge",   "4:3", 06.22, 00.69},
   &nv_bt_pal_huge_a,   &bt_pal_huge_a,     0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT},

  {{TV_SYSTEM_PAL, 800, 600, "Small",  "4:3", 14.53, 13.19},
   &nv_bt_pal_small_b,  &bt_pal_small_b,    0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_PAL, 800, 600, "Normal", "4:3", 10.81, 05.56},
   &nv_bt_pal_normal_b, &bt_pal_normal_b,   0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_PAL, 800, 600, "Large",  "4:3", 07.461, 02.083},
   &nv_bt_pal_large_b,  &bt_pal_large_b,   -8, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_PAL, 800, 600, "Huge",   "4:3", 00.039, 00.00},
   &nv_bt_pal_huge_b,   &bt_pal_huge_b,    14, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT},

  {{TV_SYSTEM_PAL, 768, 576, "Small",  "4:3", 13.122, 07.986},
   &nv_bt_pal_small_c,  &bt_pal_small_c,    0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_PAL, 768, 576, "Normal", "4:3", 09.806, 04.514},
   &nv_bt_pal_normal_c, &bt_pal_normal_c,   0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_PAL, 768, 576, "Huge", "4:3", 00.000, 00.000},
   &nv_bt_pal_huge_c, &bt_pal_huge_c,   0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},

  {{TV_SYSTEM_PAL, 800, 576, "Normal", "???", 08.83, 07.29},
   &nv_bt_pal_normal_d, &bt_pal_normal_d,   0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_PAL, 720, 576, "DVD",    "DVD", 00.208, 00.000},
   &nv_bt_pal_dvd_a,    &bt_pal_dvd_a,      4, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},

  {{TV_SYSTEM_PAL_60, 640, 480, "Small",  "4:3", 16.76, 14.81},
   &nv_bt_pal60_small_a,  &bt_pal60_small_a,    0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_PAL_60, 640, 480, "Normal",  "4:3", 10.59, 08.23},
   &nv_bt_pal60_normal_a, &bt_pal60_normal_a,   0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_PAL_60, 640, 480, "Huge",  "4:3", 01.47, 01.23},
   &nv_bt_pal60_huge_a,   &bt_pal60_huge_a,     0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},

  {{TV_SYSTEM_PAL_60, 800, 600, "Small",  "4:3", 21.62, 11.52}, 
   &nv_bt_pal60_small_b,  &bt_pal60_small_b,    0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_PAL_60, 800, 600, "Normal", "4:3", 11.90, 05.35}, 
   &nv_bt_pal60_normal_b, &bt_pal60_normal_b,   0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_PAL_60, 800, 600, "Huge",   "4:3", 07.86, 00.82},
   &nv_bt_pal60_huge_b,   &bt_pal60_huge_b,     0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},

  {{TV_SYSTEM_PAL_60, 720, 480, "DVD", "DVD", 01.245, 01.235},
   &nv_bt_pal60_dvd_a, &bt_pal60_dvd_a,   0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT_DUAL},

#ifdef CONFIG_ASYNC_BT
  {{TV_SYSTEM_PAL, 640, 480, "Small*",  "4:3", 16.56, 16.67},
   &nv_bt_async_pal_small_a,  &bt_async_pal_small_a,    0, 
    PORT_NVIDIA_PCLK_SLAVE, PORT_NVIDIA, TV_DEF_BT_DUAL},
  {{TV_SYSTEM_PAL, 640, 480, "Small$",  "4:3", 16.56, 16.67},
   &nv_bt_slave_pal_small_a,  &bt_pal_small_a,    0, 
    PORT_NVIDIA_SYNC_SLAVE, PORT_NVIDIA_SYNC_SLAVE, TV_DEF_BT_DUAL},
#endif

#ifdef CONFIG_OVERLAY_NV
  {{TV_SYSTEM_PAL, 640, 480, "Small@",  "4:3", 16.56, 16.67},
   &nv_bt_ovly_pal_small_a,  &bt_pal_small_a,    0, 
    PORT_NVIDIA, PORT_NVIDIA, TV_DEF_BT | TV_CAP_OVERLAY},
#endif

  {{TV_SYSTEM_NONE, 0, 0, NULL, NULL, 0.0, 0.0}, NULL, NULL, 0, 0},
};

/* -------- -------- */

static TVMode *modes_nv_bt   = NULL;

/*
 * count number of modes matching system; TV_SYSTEM_NONE matches all.
 */

int 
data_count_nv_bt (TVSystem system)
{
  int count;
  TVTemplateNvBt *t;

  count = 0;
  for (t = templ_nv_bt; t->spec.system != TV_SYSTEM_NONE; t++) {
    if (system == TV_SYSTEM_NONE || system == t->spec.system) count++;
  }
  return count;
}

static TVMode*
loop_create_nv_bt (TVMode *m, TVTemplateNvBt *t, 
  TVSystem convert_from, TVSystem convert_to)
{
  for (; t->spec.system != TV_SYSTEM_NONE; t++) {
    if (convert_from == TV_SYSTEM_NONE || t->spec.system == convert_from) 
    {
      m->spec = t->spec;
      if (convert_to != TV_SYSTEM_NONE) m->spec.system = convert_to;
      m->descFlags = t->descFlags;
      m->regs.portHost = t->hostFlags;
      m->regs.portEnc = t->encFlags;
      m->regs.portHost = m->regs.portEnc = PORT_NVIDIA;
      m->regs.devFlags = DEV_TELEVISION;
      if (t->descFlags & TV_CAP_DUALVIEW) 
	m->regs.devFlags = DEV_MONITOR | DEV_TELEVISION;
      if (t->descFlags & TV_CAP_OVERLAY) 
	m->regs.devFlags = DEV_TELEVISION | DEV_OVERLAY;
      m->regs.crtc.nv = *(t->nv);
      m->regs.enc.bt = *(t->bt);
      m->regs.enc.bt.hsynoffset = t->hsynoffset;
      data_mux_nv_bt (&m->regs.enc.bt);
      data_init_nomux_bt (m->spec.system, &m->regs.enc.bt);
      data_init_nv (&m->regs.crtc.nv, m->regs.portHost, m->regs.devFlags);
      switch (convert_to) 
      {
	case TV_SYSTEM_NTSC_J:
	  break; /* handled by init */
	case TV_SYSTEM_PAL_M:
	  data_ntsc_palm_bt (&m->regs.enc.bt); break;
        default: 
	  break;
      }
      m++;
    }
  }
  return m;
}

TVMode*
data_modes_nv_bt (void)
{
  int c;
  TVMode *m;

  if (modes_nv_bt) return modes_nv_bt;
  c = data_count_nv_bt (TV_SYSTEM_NONE) 
    + data_count_nv_bt (TV_SYSTEM_NTSC) /* NTSC_J */
    + data_count_nv_bt (TV_SYSTEM_NTSC) /* PAL_M */
    + 1;                                /* final zero */
  modes_nv_bt = (TVMode *) malloc (sizeof (TVMode) * c);
  
  m = modes_nv_bt;
  m = loop_create_nv_bt (m, templ_nv_bt, TV_SYSTEM_NONE, TV_SYSTEM_NONE);
  m = loop_create_nv_bt (m, templ_nv_bt, TV_SYSTEM_NTSC, TV_SYSTEM_NTSC_J);
  m = loop_create_nv_bt (m, templ_nv_bt, TV_SYSTEM_NTSC, TV_SYSTEM_PAL_M);
  m->spec.system = TV_SYSTEM_NONE;
  return modes_nv_bt;
}

/* ======== CX ================================ */

#define TV_DEF_CX (TV_CAP_MACROVISION | TV_CAP_NONINTERLACED \
                   | TV_CAP_MONOCHROME)

#define TV_DEF_CX_DUAL (TV_DEF_CX | TV_DEF_DUALVIEW)

/* -------- CX -------- NTSC -------- */

/* -------- 1024x768 3:2 -------- */

TVNvRegs cx_crt_ntsc_small_a = { 
  HDisplay    : 1024,
  HSyncStart  : 1040, /* guess */
  HSyncEnd    : 1104, /* guess */
  HTotal      : 1176,
  VDisplay    : 768,
  VSyncStart  : 845,  /* guess */
  VSyncEnd    : 847,  /* guess */
  VTotal      : 975,
  // PrivFlags   : TV_DEF_CX | TV_DEF_DUALVIEW,
  flags       : 0,
};

TVNvRegs cx_crt_ntsc_normal_a = { 
  HDisplay    : 1024,
  HSyncStart  : 1040, /* guess */
  HSyncEnd    : 1104, /* guess */
  HTotal      : 1170,
  VDisplay    : 768,
  VSyncStart  : 830, /* guess */
  VSyncEnd    : 832, /* guess */
  VTotal      : 945,
  // Privflags   : TV_DEF_CX | TV_DEF_DUALVIEW,
  flags       : 0,
};

TVNvRegs cx_crt_ntsc_tiny_a = { 
  HDisplay    : 1024,
  HSyncStart  : 1040, /* guess */
  HSyncEnd    : 1104, /* guess */
  HTotal      : 1170,
  VDisplay    : 768,
  VSyncStart  : 865, /* guess */
  VSyncEnd    : 867, /* guess */
  VTotal      : 1015,
  // Privflags   : TV_DEF_CX | TV_DEF_DUALVIEW,
  flags       : 0,
};

/* -------- 800x600 3:2 -------- */

TVNvRegs cx_crt_ntsc_small_b = { 
  HDisplay    : 800,
  HSyncStart  : 880, /* verified, Arne */
  HSyncEnd    : 936, 
  HTotal      : 1176,
  VDisplay    : 600,
  VSyncStart  : 656, 
  VSyncEnd    : 658, 
  VTotal      : 750,
  // Privflags   : TV_DEF_CX | TV_DEF_DUALVIEW,
  flags       : 0,
};

TVNvRegs cx_crt_ntsc_tiny_b = { 
  HDisplay    : 800,
  HSyncStart  : 920,  /* verified, Arne */
  HSyncEnd    : 984, 
  HTotal      : 1194, /* verified, Arne. Calc: 1170 */
  VDisplay    : 600,
  VSyncStart  : 680,  
  VSyncEnd    : 682,  
  VTotal      : 805,
  // Privflags   : TV_DEF_CX | TV_DEF_DUALVIEW,
  flags       : 0,
};

TVNvRegs cx_crt_ntsc_mini_b = { 
  HDisplay    : 800,
  HSyncStart  : 900,  /* verified, Arne */
  HSyncEnd    : 964, 
  HTotal      : 1194, /* verified, Arne. Calc: 1170 */
  VDisplay    : 600,
  VSyncStart  : 665, 
  VSyncEnd    : 667, 
  VTotal      : 770,
  // Privflags   : TV_DEF_CX | TV_DEF_DUALVIEW,
  flags       : 0,
};

/* -------- CX -------- PAL -------- */

/* -------- 1024x768 3:2 -------- */

TVNvRegs cx_crt_pal_small_a = { 
  HDisplay    : 1024,
  HSyncStart  : 1040, /* verified, Arne */
  HSyncEnd    : 1096, /* verified, Arne */
  HTotal      : 1328, /* verified, Arne. Calc: 1400 */
  VDisplay    : 768,
  VSyncStart  : 846,  /* verified, Arne. */
  VSyncEnd    : 848,  /* Arne: 846 ??? */
  VTotal      : 980,
  // Privflags   : TV_DEF_CX | TV_DEF_DUALVIEW,
  flags       : 0,
};

TVNvRegs cx_crt_pal_mini_a = { 
  HDisplay    : 1024,
  HSyncStart  : 1104, /* verified, Arne. */
  HSyncEnd    : 1184, /* verified, Arne. */
  HTotal      : 1394,
  VDisplay    : 768,
  VSyncStart  : 853, 
  VSyncEnd    : 855, 
  VTotal      : 1000,
  // Privflags   : TV_DEF_CX | TV_DEF_DUALVIEW,
  flags       : 0,
};

/* -------- CX modes -------- */

typedef struct {
  TVModeSpec spec;
  TVNvRegs *nv;
  TVCxRegs *cx;
  int hostFlags;
  int encFlags;
  int descFlags;
} TVTemplateNvCx;

TVTemplateNvCx templ_nv_cx [] = {
  {{TV_SYSTEM_NTSC, 1024, 768, "Tiny",   "4:3", 18.04, 18.11},  /* Mode 42 */
   &cx_crt_ntsc_tiny_a,   &cx_ntsc_tiny_a,   
   PORT_NVIDIA, PORT_NVIDIA, TV_DEF_CX_DUAL},
  {{TV_SYSTEM_NTSC, 1024, 768, "Small",  "4:3", 15.11, 14.81},  /* Mode 10 */
   &cx_crt_ntsc_small_a,  &cx_ntsc_small_a,  
   PORT_NVIDIA, PORT_NVIDIA, TV_DEF_CX_DUAL},
  {{TV_SYSTEM_NTSC, 1024, 768, "Normal", "4:3", 11.97, 11.93},  /* Mode 26 */
   &cx_crt_ntsc_normal_a, &cx_ntsc_normal_a, 
   PORT_NVIDIA, PORT_NVIDIA, TV_DEF_CX_DUAL},

  {{TV_SYSTEM_NTSC, 800, 600, "Tiny+",  "4:3", 19.26, 19.34},   /* Mode 34 */
   &cx_crt_ntsc_tiny_b,  &cx_ntsc_tiny_b,  
   PORT_NVIDIA, PORT_NVIDIA, TV_DEF_CX_DUAL},
  {{TV_SYSTEM_NTSC, 800, 600, "Mini+",  "4:3", 15.59, 15.64},   /* Mode 40 */
   &cx_crt_ntsc_mini_b,  &cx_ntsc_mini_b,  
   PORT_NVIDIA, PORT_NVIDIA, TV_DEF_CX_DUAL},
  {{TV_SYSTEM_NTSC, 800, 600, "Small+", "4:3", 13.79, 13.58},   /* Mode 18 */
   &cx_crt_ntsc_small_b, &cx_ntsc_small_b, 
   PORT_NVIDIA, PORT_NVIDIA, TV_DEF_CX_DUAL},

  {{TV_SYSTEM_PAL, 1024, 768, "Mini",  "4:3", 16.20, 16.67},    /* Mode 43 */
   &cx_crt_pal_mini_a, &cx_pal_mini_a,   
   PORT_NVIDIA, PORT_NVIDIA, TV_DEF_CX_DUAL},
  {{TV_SYSTEM_PAL, 1024, 768, "Small", "4:3", 13.44, 14.24},    /* Mode 11 */
   &cx_crt_pal_small_a, &cx_pal_small_a, 
   PORT_NVIDIA, PORT_NVIDIA, TV_DEF_CX_DUAL},

  {{TV_SYSTEM_NONE, 0, 0, NULL, NULL, 0.0, 0.0}, NULL, NULL, 0},
};

/* -------- -------- */

static TVMode*
modes_nv_cx = NULL;

/*
 * count number of modes matching system; TV_SYSTEM_NONE matches all.
 */

int 
data_count_nv_cx (TVTemplateNvCx *t, TVSystem system)
{
  int count;

  count = 0;
  for (; t->spec.system != TV_SYSTEM_NONE; t++) {
    if (system == TV_SYSTEM_NONE || system == t->spec.system) count++;
  }
  return count;
}

/*
 * Create CX modes from both BT and CX templates.
 */

static TVMode*
loop_create_nv_btcx (TVMode *m, TVTemplateNvBt *t, 
  TVSystem convert_from, TVSystem convert_to)
{
  for (; t->spec.system != TV_SYSTEM_NONE; t++) {
    if (convert_from == TV_SYSTEM_NONE || t->spec.system == convert_from) 
    {
      m->spec = t->spec;
      if (convert_to != TV_SYSTEM_NONE) m->spec.system = convert_to;
      m->descFlags = t->descFlags;
      m->regs.portHost = t->hostFlags;
      m->regs.portEnc = t->encFlags;
      m->regs.portHost = m->regs.portEnc = PORT_NVIDIA;
      m->regs.devFlags = (t->descFlags & TV_CAP_DUALVIEW) ? 
	(DEV_MONITOR | DEV_TELEVISION) : (DEV_TELEVISION);
      m->regs.crtc.nv = *(t->nv);
      m->regs.enc.cx.bt = *(t->bt);
      m->regs.enc.cx.bt.hsynoffset = t->hsynoffset;
      /* does data_mux_nv_bt */
      data_init_cx (m->spec.system, &m->regs.enc.cx, TRUE);
      data_init_nv (&m->regs.crtc.nv, m->regs.portHost, m->regs.devFlags);
      switch (convert_to) 
      {
	case TV_SYSTEM_NTSC_J:
	  break; /* handled by init */
	case TV_SYSTEM_PAL_M:
	  data_ntsc_palm_bt (&m->regs.enc.cx.bt); break;
	case TV_SYSTEM_SECAM: 
	  data_secam_cx (&m->regs.enc.cx); break;
        default: 
	  break;
      }
      m++;
    }
  }
  return m;
}

static TVMode*
loop_create_nv_cx (TVMode *m, TVTemplateNvCx *t, 
  TVSystem convert_from, TVSystem convert_to)
{
  for (; t->spec.system != TV_SYSTEM_NONE; t++) {
    if (convert_from == TV_SYSTEM_NONE || t->spec.system == convert_from) 
    {
      m->spec = t->spec;
      if (convert_to != TV_SYSTEM_NONE) m->spec.system = convert_to;
      m->descFlags = t->descFlags;
      m->regs.portHost = t->hostFlags;
      m->regs.portEnc = t->encFlags;
      m->regs.devFlags = (t->descFlags & TV_CAP_DUALVIEW) ? 
	(DEV_MONITOR | DEV_TELEVISION) : (DEV_TELEVISION);
      m->regs.crtc.nv = *(t->nv);
      m->regs.enc.cx = *(t->cx);
      /* does data_mux_nv_bt */
      data_init_cx (m->spec.system, &m->regs.enc.cx, FALSE);
      data_init_nv (&m->regs.crtc.nv, m->regs.portHost, m->regs.devFlags);
      switch (convert_to) 
      {
	case TV_SYSTEM_NTSC_J:
	  break; /* handled by init */
	case TV_SYSTEM_PAL_M:
	  data_ntsc_palm_bt (&m->regs.enc.cx.bt); break;
	case TV_SYSTEM_SECAM: 
	  data_secam_cx (&m->regs.enc.cx); break;
        default: 
	  break;
      }
      m++;
    }
  }
  return m;
}

TVMode 
*data_modes_nv_cx (void)
{
  int c;
  TVMode *m;

  if (modes_nv_cx) return modes_nv_cx;
  c = data_count_nv_bt (TV_SYSTEM_NONE)
    + data_count_nv_bt (TV_SYSTEM_PAL)                 /* SECAM */
    + data_count_nv_bt (TV_SYSTEM_NTSC)                /* NTSC_J */
    + data_count_nv_bt (TV_SYSTEM_NTSC)                /* PAL_M */
    + data_count_nv_cx (templ_nv_cx, TV_SYSTEM_NONE)
    + data_count_nv_cx (templ_nv_cx, TV_SYSTEM_PAL)    /* SECAM */
    + data_count_nv_cx (templ_nv_cx, TV_SYSTEM_NTSC)   /* NTSC_J */
    + data_count_nv_cx (templ_nv_cx, TV_SYSTEM_NTSC)   /* PAL_M */
    + 1;                                               /* final zero */
  modes_nv_cx = (TVMode *) malloc (sizeof (TVMode) * c);
  
  m = modes_nv_cx;
  m = loop_create_nv_btcx (m, templ_nv_bt, TV_SYSTEM_NONE, TV_SYSTEM_NONE);
  m = loop_create_nv_btcx (m, templ_nv_bt, TV_SYSTEM_PAL,  TV_SYSTEM_SECAM);
  m = loop_create_nv_btcx (m, templ_nv_bt, TV_SYSTEM_NTSC, TV_SYSTEM_NTSC_J);
  m = loop_create_nv_btcx (m, templ_nv_bt, TV_SYSTEM_NTSC, TV_SYSTEM_PAL_M);
  m = loop_create_nv_cx (m, templ_nv_cx, TV_SYSTEM_NONE, TV_SYSTEM_NONE);
  m = loop_create_nv_cx (m, templ_nv_cx, TV_SYSTEM_PAL,  TV_SYSTEM_SECAM);
  m = loop_create_nv_cx (m, templ_nv_cx, TV_SYSTEM_NTSC, TV_SYSTEM_NTSC_J);
  m = loop_create_nv_cx (m, templ_nv_cx, TV_SYSTEM_NTSC, TV_SYSTEM_PAL_M);
  m->spec.system = TV_SYSTEM_NONE;
  return modes_nv_cx;
}

/* ======== CH ================================ */

#define TV_DEF_CH (TV_CAP_MONOCHROME | TV_CAP_COLORFIX)

/* -------- CH1 -------- NTSC -------- */

/* -------- 640x480 -------- */

TVNvRegs nv_ch1_ntsc_huge_a = { 
  HDisplay    : 640,
  HSyncStart  : 664-8,
  HSyncEnd    : 736-8,
  HTotal      : 784, /* Mode 16 ch: 784 */
  VDisplay    : 480,
  VSyncStart  : 486,
  VSyncEnd    : 488,
  VTotal      : 525, /* Mode 16 ch: 525 */
  flags	      : 0,
};

TVNvRegs nv_ch1_ntsc_small_a = { 
  HDisplay    : 640,
  HSyncStart  : 664-8,
  HSyncEnd    : 736-8,
  HTotal      : 784, /* Mode 17 ch: 784 */
  VDisplay    : 480,
  VSyncStart  : 522,
  VSyncEnd    : 524,
  VTotal      : 600, /* Mode 17 ch: 600 */
  flags	      : 0,
};

TVNvRegs nv_ch1_ntsc_tiny_a = { 
  HDisplay    : 640,
  HSyncStart  : 672-8,
  HSyncEnd    : 744-8,
  HTotal      : 800, /* Mode 18 ch: 800 */
  VDisplay    : 480,
  VSyncStart  : 538,
  VSyncEnd    : 540,
  VTotal      : 630, /* Mode 18 ch: 630 */
  flags	      : 0,
};

/* -------- 800x600 -------- */

TVNvRegs nv_ch1_ntsc_huge_b = { 
  HDisplay    : 800,
  HSyncStart  : 856-8,
  HSyncEnd    : 920-8,
  HTotal      : 1040, /* Mode 22 ch: 1040 */
  VDisplay    : 600,
  VSyncStart  : 600,
  VSyncEnd    : 602,
  VTotal      : 630,  /* Mode 22 ch: 630 */
  flags	      : 0,
};

TVNvRegs nv_ch1_ntsc_large_b = { 
  HDisplay    : 800,
  HSyncStart  : 880-8,
  HSyncEnd    : 968-8,
  HTotal      : 1040, /* Mode 23 ch: 1040 */
  VDisplay    : 600,
  VSyncStart  : 626,
  VSyncEnd    : 628,
  VTotal      : 700,  /* Mode 23 ch: 700 */
  flags	      : 0,
};

TVNvRegs nv_ch1_ntsc_small_b = { 
  HDisplay    : 800,
  HSyncStart  : 896-8,
  HSyncEnd    : 960-8,
  HTotal      : 1064, /* Mode 24 ch: 1064 */
  VDisplay    : 600,
  VSyncStart  : 652,
  VSyncEnd    : 654,
  VTotal      : 750,  /* Mode 24 ch: 750 */
  flags	      : 0,
};

/* -------- 640x400 -------- */

TVNvRegs nv_ch1_ntsc_huge_c = { /* FIXME: Doesn't work */
  HDisplay    : 640,
  HSyncStart  : 704-8,
  HSyncEnd    : 768-8,
  HTotal      : 840, /* Mode 10 ch: 840 */
  VDisplay    : 400,
  VSyncStart  : 410,
  VSyncEnd    : 411,
  VTotal      : 420, /* Mode 10 ch: 420 */
  flags	      : 0,
};

TVNvRegs nv_ch1_ntsc_small_c = { 
  HDisplay    : 640,
  HSyncStart  : 696-8,
  HSyncEnd    : 760-8,
  HTotal      : 840, /* Mode 11 ch: 840 */
  VDisplay    : 400,
  VSyncStart  : 444,
  VSyncEnd    : 446,
  VTotal      : 525, /* Mode 11 ch: 525 */
  flags	      : 0,
};

TVNvRegs nv_ch1_ntsc_tiny_c = { 
  HDisplay    : 640,
  HSyncStart  : 696-8,
  HSyncEnd    : 768-8,
  HTotal      : 840, /* Mode 12 ch: 840 */
  VDisplay    : 400,
  VSyncStart  : 480,
  VSyncEnd    : 482,
  VTotal      : 600, /* Mode 12 ch: 600 */
  flags	      : 0,
};

/* -------- 720x400 -------- */

TVNvRegs nv_ch1_ntsc_huge_d = { 
  HDisplay    : 720,
  HSyncStart  : 784-8,
  HSyncEnd    : 864-8,
  HTotal      : 945, /* Mode 6 ch: 945 */
  VDisplay    : 400,
  VSyncStart  : 404,
  VSyncEnd    : 405,
  VTotal      : 420, /* Mode 6 ch: 420 */
  flags	      : 0,
};

TVNvRegs nv_ch1_ntsc_small_d = { 
  HDisplay    : 720,
  HSyncStart  : 784-8,
  HSyncEnd    : 864-8,
  HTotal      : 936, /* Mode 7 ch: 936 */
  VDisplay    : 400,
  VSyncStart  : 450,
  VSyncEnd    : 452,
  VTotal      : 525, /* Mode 7 ch: 525 */
  flags	      : 0,
};

/* -------- 512x384 -------- */

TVNvRegs nv_ch1_ntsc_large_e = { 
  HDisplay    : 512,
  HSyncStart  : 616-8,
  HSyncEnd    : 688-8,
  HTotal      : 800, /* Mode 2 ch: 800 */
  VDisplay    : 384,
  VSyncStart  : 396,
  VSyncEnd    : 398,
  VTotal      : 420, /* Mode 2 ch: 420 */
  flags	      : 0,
};

TVNvRegs nv_ch1_ntsc_small_e = { 
  HDisplay    : 512,
  HSyncStart  : 608-8,
  HSyncEnd    : 672-8,
  HTotal      : 784, /* Mode 3 ch: 784 */
  VDisplay    : 384,
  VSyncStart  : 434,
  VSyncEnd    : 436,
  VTotal      : 525, /* Mode 3 ch: 525 */
  flags	      : 0,
};

/* -------- DVD convenience modes -------- */

TVNvRegs nv_ch1_ntsc_large_f = { /* test */
  HDisplay    : 768,
  HSyncStart  : 856-8,
  HSyncEnd    : 952-8,
  HTotal      : 1040, /* Mode 23 ch: 1040 */
  VDisplay    : 576,
  VSyncStart  : 618,
  VSyncEnd    : 620,
  VTotal      : 700,  /* Mode 23 ch: 700 */
  flags	      : 0,
};

TVNvRegs nv_ch1_ntsc_large_g = { /* test */
  HDisplay    : 800,
  HSyncStart  : 880-8,
  HSyncEnd    : 968-8,
  HTotal      : 1040, /* Mode 23 ch: 1040 */
  VDisplay    : 450,
  VSyncStart  : 550,
  VSyncEnd    : 552,
  VTotal      : 700,  /* Mode 23 ch: 700 */
  flags	      : 0,
};

/* -------- 720x{576,480} DVD -------- */

/* Note: Not used yet, crt values not tested */

TVNvRegs nv_ch1_ntsc_interl_dvd = { 
  HDisplay    : 720,
  HSyncStart  : 752-8,
  HSyncEnd    : 768-8,
  HTotal      : 858, /* Mode 26 ch: 858 */
  VDisplay    : 480,
  VSyncStart  : 500,
  VSyncEnd    : 501,
  VTotal      : 525, /* Mode 26 ch: 525 */
  flags	      : 0,
};

/* -------- CH -------- PAL -------- */

/* -------- 640x480 -------- */

/* FIXME: crt_pal_large_a -- no dualview, not in tvcc either */

TVNvRegs nv_ch1_pal_huge_a = { 
  HDisplay    : 640,
  HSyncStart  : 688-8,
  HSyncEnd    : 760-8,
  HTotal      : 760, /* Mode 13 ch: 840 */
  VDisplay    : 480,
  VSyncStart  : 532,
  VSyncEnd    : 534,
  VTotal      : 574, /* Mode 13 ch: 500 */
  flags	      : 0,
};

TVNvRegs nv_ch1_pal_small_a = { 
  HDisplay    : 640,
  HSyncStart  : 688-8,
  HSyncEnd    : 760-8,
  HTotal      : 760, /* Mode 14 ch: 840 */
  VDisplay    : 480,
  VSyncStart  : 532,
  VSyncEnd    : 534,
  VTotal      : 574, /* Mode 14 ch: 625 */
  flags	      : 0,
};

TVNvRegs nv_ch1_pal_tiny_a = { 
  HDisplay    : 640,
  HSyncStart  : 688-8,
  HSyncEnd    : 760-8,
  HTotal      : 760, /* Mode 15 ch: 840 */
  VDisplay    : 480,
  VSyncStart  : 590,
  VSyncEnd    : 592,
  VTotal      : 750, /* Mode 15 ch: 750 */
  flags	      : 0,
};

/* -------- 800x600 -------- */

/* FIXME: huge has significant distortion on right hand side */

TVNvRegs nv_ch1_pal_huge_b = { 
  HDisplay    : 800,
  HSyncStart  : 808-8,
  HSyncEnd    : 872-8,
  HTotal      : 944, /* Mode 19 ch: 944 */
  VDisplay    : 600,
  VSyncStart  : 600,
  VSyncEnd    : 601,
  VTotal      : 635, /* Mode 19 ch: 625 */
  flags	      : 0,
};

TVNvRegs nv_ch1_pal_small_b = { 
  HDisplay    : 800,
  HSyncStart  : 824-8,
  HSyncEnd    : 888-8,
  HTotal      : 960, /* Mode 20 ch: 960 */
  VDisplay    : 600,
  VSyncStart  : 656,
  VSyncEnd    : 658,
  VTotal      : 750, /* Mode 20 ch: 750 */
  flags	      : 0,
};

TVNvRegs nv_ch1_pal_tiny_b = { 
  HDisplay    : 800,
  HSyncStart  : 840-8,
  HSyncEnd    : 904-8,
  HTotal      : 936, /* Mode 21 ch: 936 */
  VDisplay    : 600,
  VSyncStart  : 709,
  VSyncEnd    : 711,
  VTotal      : 836, /* Mode 21 ch: 836 */
  flags	      : 0,
};

/* Note: Mode 21 has 627 PAL lines instead of 625 */

/* -------- 640x400 -------- */

TVNvRegs nv_ch1_pal_small_c = { 
  HDisplay    : 640,
  HSyncStart  : 824-8,
  HSyncEnd    : 888-8,
  HTotal      : 1000, /* Mode 8 ch: 1000 */
  VDisplay    : 400,
  VSyncStart  : 440,
  VSyncEnd    : 442,
  VTotal      : 500,  /* Mode 8 ch: 500 */
  flags	      : 0,
};

TVNvRegs nv_ch1_pal_tiny_c = { 
  HDisplay    : 640,
  HSyncStart  : 760-8,
  HSyncEnd    : 824-8,
  HTotal      : 936, /* Mode 9 ch: 1008 */
  VDisplay    : 400,
  VSyncStart  : 496,
  VSyncEnd    : 498,
  VTotal      : 625, /* Mode 9 ch: 625 */
  flags	      : 0,
};

/* -------- 720x400 -------- */

TVNvRegs nv_ch1_pal_small_d = { 
  HDisplay    : 720,
  HSyncStart  : 824-8,
  HSyncEnd    : 888-8,
  HTotal      : 1125, /* Mode 4 ch: 1125 */
  VDisplay    : 400,
  VSyncStart  : 437,
  VSyncEnd    : 439,
  VTotal      : 500,  /* Mode 4 ch: 500 */
  flags	      : 0,
};

TVNvRegs nv_ch1_pal_tiny_d = { 
  HDisplay    : 720,
  HSyncStart  : 840-8,
  HSyncEnd    : 904-8,
  HTotal      : 1116, /* Mode 5 ch: 1116 */
  VDisplay    : 400,
  VSyncStart  : 498,
  VSyncEnd    : 500,
  VTotal      : 625,  /* Mode 5 ch: 625 */
  flags	      : 0,
};

/* -------- 512x384 -------- */

TVNvRegs nv_ch1_pal_small_e = { 
  HDisplay    : 512,
  HSyncStart  : 640-8,
  HSyncEnd    : 704-8,
  HTotal      : 840,  /* Mode 0 ch: 840 */
  VDisplay    : 384,
  VSyncStart  : 428,
  VSyncEnd    : 430,
  VTotal      : 500,  /* Mode 0 ch: 500 */
  flags	      : 0,
};

TVNvRegs nv_ch1_pal_tiny_e = { 
  HDisplay    : 512,
  HSyncStart  : 624-8,
  HSyncEnd    : 704-8,
  HTotal      : 840,  /* Mode 1 ch: 840 */
  VDisplay    : 384,
  VSyncStart  : 498,
  VSyncEnd    : 500,
  VTotal      : 625,  /* Mode 1 ch: 625 */
  flags	      : 0,
};

/* -------- DVD convenience modes -------- */

TVNvRegs nv_ch1_pal_large_f = { /* test FIXME: SyncStart ok ?? */
  HDisplay    : 768,
  HSyncStart  : 800-8,
  HSyncEnd    : 872-8,
  HTotal      : 944, /* Mode 19 ch: 944 */
  VDisplay    : 576,
  VSyncStart  : 592,
  VSyncEnd    : 594,
  VTotal      : 635, /* Mode 19 ch: 625 */
  flags	      : 0,
};

TVNvRegs nv_ch1_pal_large_g = { /* test */
  HDisplay    : 800,
  HSyncStart  : 808-8,
  HSyncEnd    : 872-8,
  HTotal      : 944, /* Mode 19 ch: 944 */
  VDisplay    : 450,
  VSyncStart  : 530,
  VSyncEnd    : 532,
  VTotal      : 635, /* Mode 19 ch: 625 */
  flags	      : 0,
};

TVNvRegs nv_ch1_pal_small_f = { /* test */
  HDisplay    : 768,
  HSyncStart  : 808-8,
  HSyncEnd    : 880-8,
  HTotal      : 960, /* Mode 20 ch: 960 */
  VDisplay    : 576,
  VSyncStart  : 642,
  VSyncEnd    : 644,
  VTotal      : 750, /* Mode 20 ch: 750 */
  flags	      : 0,
};

TVNvRegs nv_ch1_pal_small_g = { /* test */
  HDisplay    : 800,
  HSyncStart  : 824-8,
  HSyncEnd    : 888-8,
  HTotal      : 960, /* Mode 20 ch: 960 */
  VDisplay    : 450,
  VSyncStart  : 582,
  VSyncEnd    : 584,
  VTotal      : 750, /* Mode 20 ch: 750 */
  flags	      : 0,
};

/* -------- 720x{576,480} DVD -------- */

TVNvRegs nv_ch1_pal_interl_dvd = { 
  HDisplay    : 720,
  HSyncStart  : 752-8,
  HSyncEnd    : 768-8,
  HTotal      : 864, /* Mode 25 ch: 864 */
  VDisplay    : 576,
  VSyncStart  : 600,
  VSyncEnd    : 602,
  VTotal      : 625, /* Mode 25 ch: 625 */
  flags	      : 0,
};

/* -------- CH2 -------- NTSC -------- */

TVNvRegs nv_ch2_ntsc_huge_c  = { /* Mode 19 */
  HDisplay   : 720,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 882,
  VDisplay   : 480,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 525,
  flags	     : 0,
};

TVNvRegs nv_ch2_ntsc_small_c = { /* Mode 20 */
  HDisplay   : 720,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 882,
  VDisplay   : 480,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 600,
  flags	     : 0,
};

TVNvRegs nv_ch2_ntsc_tiny_c  = { /* Mode 21 */
  HDisplay   : 720,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 900,
  VDisplay   : 480,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 630,
  flags	     : 0,
};

TVNvRegs nv_ch2_ntsc_large_b = { /* Mode 28 */
  HDisplay   : 800,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 1040,
  VDisplay   : 600,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 700,
  flags	     : 0,
};

TVNvRegs nv_ch2_ntsc_small_b = { /* Mode 29 */
  HDisplay   : 800,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 1064,
  VDisplay   : 600,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 750,
  flags	     : 0,
};

TVNvRegs nv_ch2_ntsc_tiny_b  = { /* Mode 30 */
  HDisplay   : 800,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 1040,
  VDisplay   : 600,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 840,
  flags	     : 0,
};

TVNvRegs nv_ch2_ntsc_huge_a  = { /* Mode 34 */
  HDisplay   : 1024,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 1160,
  VDisplay   : 768,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 840,
  flags	     : 0,
};

TVNvRegs nv_ch2_ntsc_large_a = { /* Mode 35 */
  HDisplay   : 1024,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 1160,
  VDisplay   : 768,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 945,
  flags	     : 0,
};

TVNvRegs nv_ch2_ntsc_tiny_a  = { /* Mode 36 */
  HDisplay   : 1024,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 1168,
  VDisplay   : 768,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 1050,
  flags	     : 0,
};

TVNvRegs nv_ch2_ntsc_dvd     = { /* Mode 38 */
  HDisplay   : 720,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 858,
  VDisplay   : 480,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 525,
  flags	     : 0,
};

/* -------- CH2 -------- PAL -------- */

TVNvRegs nv_ch2_pal_huge_d   = { /* Mode 22 */
  HDisplay   : 720,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 882,
  VDisplay   : 576,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 625,
  flags	     : 0,
};

TVNvRegs nv_ch2_pal_small_d  = { /* Mode 23 */
  HDisplay   : 720,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 900,
  VDisplay   : 576,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 750,
  flags	     : 0,
};

TVNvRegs nv_ch2_pal_tiny_d   = { /* Mode 24 */
  HDisplay   : 720,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 900,
  VDisplay   : 576,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 875,
  flags	     : 0,
};
 
TVNvRegs nv_ch2_pal_huge_b   = { /* Mode 25 */
  HDisplay   : 800,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 944,
  VDisplay   : 600,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 625,
  flags	     : 0,
};

TVNvRegs nv_ch2_pal_small_b  = { /* Mode 26 */
  HDisplay   : 800,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 960,
  VDisplay   : 600,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 750,
  flags	     : 0,
};

TVNvRegs nv_ch2_pal_tiny_b   = { /* Mode 27 */
  HDisplay   : 800,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 960,
  VDisplay   : 600,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 875,
  flags	     : 0,
};

TVNvRegs nv_ch2_pal_huge_a   = { /* Mode 31 */
  HDisplay   : 1024,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 1400,
  VDisplay   : 768,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 875,
  flags	     : 0,
};

TVNvRegs nv_ch2_pal_small_a  = { /* Mode 32 */
  HDisplay   : 1024,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 1400,
  VDisplay   : 768,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 1000,
  flags	     : 0,
};

TVNvRegs nv_ch2_pal_tiny_a   = { /* Mode 33 */
  HDisplay   : 1024,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 1400,
  VDisplay   : 768,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 1125,
  flags	     : 0,
};

TVNvRegs nv_ch2_pal_dvd      = { /* Mode 37 */
  HDisplay   : 720,
  HSyncStart : 0,
  HSyncEnd   : 0,
  HTotal     : 864,
  VDisplay   : 576,
  VSyncStart : 0,
  VSyncEnd   : 0,
  VTotal     : 625,
  flags	     : 0,
};

/* -------- CH1/NV -------- Slave modes -------- */

/*
NTSC
   000001a4 0000020d 18 ; 320x200 ->  420 525 24 Mode 11 doublepix
   00000188 00000258 0d ; 320x240 ->  392 600 13 Mode 17 doublepix
   00000348 0000020d 03 ; 640x400 ->  840 525 03 Mode 11
   00000310 00000258 0d ; 640x480 ->  784 600 13 Mode 17
   00000428 000002ee 00 ; 720x480 -> 1064 750  0 Mode (24)
   00000410 00000276 00 ; 720x576 -> 1040 630  0 Mode (22)
   00000428 000002ee 0d ; 800x600 -> 1064 750 13 Mode 24
   		     	  
PAL		     	
   000001f4 000001f4 06 ; 320x200 ->  500 500  6 Mode 8 doublepix
   000001a4 00000271 08 ; 320x240 ->  420 625  8 Mode 1,14 ?
   000003e8 000001f4 05 ; 640x400 -> 1000 500  5 Mode 8
   00000348 00000271 05 ; 640x480 ->  840 625  5 Mode 14
   000003c0 000002ee 05 ; 720x480 ->  960 750  5 Mode (20)
   000003b0 00000271 00 ; 720x576 ->  944 625  0 Mode 19
   000003c0 000002ee 05 ; 800x600 ->  960 750  5 Mode 20

Mode  0:  512 384    840 500
   525, 800   0.833333
   560, 750   0.888889
   600, 700   0.952381
   625, 672   0.992063
   672, 625   1.06667
   700, 600   1.11111
   750, 560   1.19048
   800, 525   1.26984
   840, 500   1.33333
Mode  1:  512 384    840 625
   525, 1000   0.833333
   600, 875   0.952381
   625, 840   0.992063
   700, 750   1.11111
   750, 700   1.19048
   840, 625   1.33333
Mode  2:  512 384    800 420
   525, 640   0.875
   560, 600   0.933333
   600, 560   1.0
   640, 525   1.06667
   672, 500   1.12
   700, 480   1.16667
   750, 448   1.25
   800, 420   1.33333
Mode  3:  512 384    784 525
   525, 784   0.892857
   560, 735   0.952381
   588, 700   1.0
   600, 686   1.02041
   686, 600   1.16667
   700, 588   1.19048
   735, 560   1.25
   784, 525   1.33333
Mode  4:  720 400   1125 500
   750, 750   1.2
   900, 625   1.44
  1125, 500   1.8
Mode  5:  720 400   1116 625
   750, 930   1.20968
   775, 900   1.25
   900, 775   1.45161
   930, 750   1.5
  1116, 625   1.8
Mode  6:  720 400    945 420
   735, 540   1.4
   756, 525   1.44
   810, 490   1.54286
   882, 450   1.68
   900, 441   1.71429
   945, 420   1.8
Mode  7:  720 400    936 525
   728, 675   1.4
   756, 650   1.45385
   780, 630   1.5
   819, 600   1.575
   840, 585   1.61538
   900, 546   1.73077
   910, 540   1.75
   936, 525   1.8
Mode  8:  640 400   1000 500
   800, 625   1.28
  1000, 500   1.6
Mode  9:  640 400   1008 625
   700, 900   1.11111
   720, 875   1.14286
   750, 840   1.19048
   840, 750   1.33333
   875, 720   1.38889
   900, 700   1.42857
  1000, 630   1.5873
  1008, 625   1.6
Mode 10:  640 400    840 420
   672, 525   1.28
   700, 504   1.33333
   720, 490   1.37143
   735, 480   1.4
   784, 450   1.49333
   800, 441   1.52381
   840, 420   1.6
Mode 11:  640 400    840 525
   700, 630   1.33333
   735, 600   1.4
   750, 588   1.42857
   840, 525   1.6
Mode 12:  640 400    840 600
   672, 750   1.28
   700, 720   1.33333
   720, 700   1.37143
   750, 672   1.42857
   800, 630   1.52381
   840, 600   1.6
Mode 13:  640 480    840 500
   672, 625   1.06667
   700, 600   1.11111
   750, 560   1.19048
   800, 525   1.26984
   840, 500   1.33333
Mode 14:  640 480    840 625
   700, 750   1.11111
   750, 700   1.19048
   840, 625   1.33333
Mode 15:  640 480    840 750
   700, 900   1.11111
   720, 875   1.14286
   750, 840   1.19048
   840, 750   1.33333
Mode 16:  640 480    784 525
   686, 600   1.16667
   700, 588   1.19048
   735, 560   1.25
   784, 525   1.33333
Mode 17:  640 480    784 600
   640, 735   1.08844
   672, 700   1.14286
   700, 672   1.19048
   735, 640   1.25
   784, 600   1.33333
Mode 18:  640 480    800 630
   672, 750   1.12
   700, 720   1.16667
   720, 700   1.2
   750, 672   1.25
   800, 630   1.33333
Mode 19:  800 600    944 625
   944, 625   1.33333
Mode 20:  800 600    960 750
   800, 900   1.11111
   900, 800   1.25
   960, 750   1.33333
Mode 21:  800 600    936 836
   836, 936   1.19088
   858, 912   1.22222
   912, 858   1.29915
   936, 836   1.33333
Mode 22:  800 600   1040 630
   800, 819   1.02564
   819, 800   1.05
   840, 780   1.07692
   900, 728   1.15385
   910, 720   1.16667
   936, 700   1.2
   975, 672   1.25
  1008, 650   1.29231
  1040, 630   1.33333
Mode 23:  800 600   1040 700
   800, 910   1.02564
   832, 875   1.06667
   875, 832   1.12179
   910, 800   1.16667
  1000, 728   1.28205
  1040, 700   1.33333
Mode 24:  800 600   1064 750
   840, 950   1.05263
   875, 912   1.09649
   912, 875   1.14286
   950, 840   1.19048
  1000, 798   1.25313
  1050, 760   1.31579
  1064, 750   1.33333



*/

TVNvRegs nv_ch1_ntsc_slave_a = { /* Mode 17 small_a */
  HDisplay    : 640,
  HSyncStart  : 664-8,
  HSyncEnd    : 736-8,
  HTotal      : 784, /* Mode 17 ch: 784 */
  VDisplay    : 480,
  VSyncStart  : 522,
  VSyncEnd    : 524,
  VTotal      : 600, /* Mode 17 ch: 600 */
  slave : {
    HTotal     : 784,
    HSyncStart : 1,
    HSyncEnd   : 64,
    VTotal     : 600,
    VSyncStart : 1,
    VSyncEnd   : 16,
    Unknown    : 13,
  }
};

TVChRegs ch1_ntsc_slave_b = { /* Mode 24: 800x600 7:10  */
  dmr_ir  : 4,
  dmr_vs  : 1,
  dmr_sr  : 5,
  sav     : 142,  
  hpr     : 60,   
  vpr     : 0,    
  pll_m   : 89,
  pll_n   : 302,
  pllcap  : 0,
  dacg    : 0,
  mode    : 24,
};

TVNvRegs nv_ch1_ntsc_slave_b = { /* Mode 24 small_b */
  HDisplay    : 800,
  HSyncStart  : 864,
  HSyncEnd    : 968,
  HTotal      : 984, /* Mode 24 ch: 1064 */
  VDisplay    : 600,
  VSyncStart  : 602,
  VSyncEnd    : 605,
  VTotal      : 698,  /* Mode 24 ch: 750 */
  slave : {
    HTotal     : 1064,
    HSyncStart : 1,
    HSyncEnd   : 64,
    VTotal     : 750,
    VSyncStart : 1,
    VSyncEnd   : 16,
    Unknown    : 13,
  }
};

TVNvRegs nv_ch1_pal_slave_a = { /* Mode 14 small_a */
  HDisplay    : 640,
  HSyncStart  : 688-8,
  HSyncEnd    : 760-8,
  HTotal      : 760, /* Mode 14 ch: 840 */
  VDisplay    : 480,
  VSyncStart  : 532,
  VSyncEnd    : 534,
  VTotal      : 574, /* Mode 14 ch: 625 */
  slave : {
    HTotal     : 840,
    HSyncStart : 1,
    HSyncEnd   : 64,
    VTotal     : 625,
    VSyncStart : 1,
    VSyncEnd   : 16,
    Unknown    : 5,
  }
};

TVChRegs ch1_pal_slave_b = { /* Mode 20: 800x600 5:6  */
  dmr_ir  : 4,
  dmr_vs  : 0,
  dmr_sr  : 3,
  sav     : 126, 
  hpr     : 64, 
  vpr     : 0,  
  pll_m   : 33,
  pll_n   : 86,
  pllcap  : 0, /* docs: 1 !! */
  mode    : 20,
};

TVNvRegs nv_ch1_pal_slave_b = { /* Mode 20 small_b */
  HDisplay    : 800,
  HSyncStart  : 840,
  HSyncEnd    : 968,
  HTotal      : 936,
  VDisplay    : 600,
  VSyncStart  : 602,
  VSyncEnd    : 605,
  VTotal      : 698,
  slave : {
    HTotal     : 960,
    HSyncStart : 1,
    HSyncEnd   : 64,
    VTotal     : 750,
    VSyncStart : 1,
    VSyncEnd   : 16,
    Unknown    : 5,
  }
};

/* -------- NV -------- modelist -------- */

/* No Dualview for:

PAL  640x400 T  Mode 9
PAL  720x400 T  Mode 5

NTSC 720x400 L  Mode 6
NTSC 512x384 L  Mode 2

*/

/* FIXME: ch2 modes for 800x600/768x576 are different. New modes
   have 720x480, 720x576, 800x600, 1024x768. So make 3 templates. */

/* FIXME: Calculate correct voc and hoc values. At the moment, hoc = voc,
   and the voc values seem way off for some reason. */

typedef struct {
  TVModeSpec spec;
  TVNvRegs *nv;
  TVChRegs *ch;
  unsigned long *fsci;
  int portFlags;
  int descFlags;
} TVTemplateNvCh;

/* For the PAL-X convenience mode, FSCI is sometimes the same?? 
   This is now ignored -- the convenience modes get modified as well. 
*/

TVTemplateNvCh templ_nv_ch1_50hz [] = { /* 50 Hz: PAL, PAL-N, PAL-X */

  {{TV_SYSTEM_PAL, 800, 600, "Tiny",   "4:3", 21.875, 21.875}, /* Mode 21 */
   &nv_ch1_pal_tiny_b,  &ch1_pal_tiny_b,  ch1_fsci_pal_tiny_b,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},
  {{TV_SYSTEM_PAL, 800, 600, "Small",  "4:3", 13.194, 13.194}, /* Mode 20 */
   &nv_ch1_pal_small_b, &ch1_pal_small_b, ch1_fsci_pal_small_b,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},
  {{TV_SYSTEM_PAL, 800, 600, "Huge",   "4:3", -4.167, -4.167}, /* Mode 19 */
   &nv_ch1_pal_huge_b, &ch1_pal_huge_b, ch1_fsci_pal_huge_b,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},  
  {{TV_SYSTEM_PAL, 640, 480, "Tiny",   "4:3", 30.556, 30.556}, /* Mode 15 */
   &nv_ch1_pal_tiny_a,  &ch1_pal_tiny_a,  ch1_fsci_pal_tiny_a,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},    
  {{TV_SYSTEM_PAL, 640, 480, "Small",  "4:3", 16.667, 16.667}, /* Mode 14 */
   &nv_ch1_pal_small_a, &ch1_pal_small_a, ch1_fsci_pal_small_a,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},  
  {{TV_SYSTEM_PAL, 640, 480, "Huge",   "4:3", -4.167, -4.167}, /* Mode 13 */
   &nv_ch1_pal_huge_a, &ch1_pal_huge_a, ch1_fsci_pal_huge_a,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},  
  {{TV_SYSTEM_PAL, 640, 400, "Tiny",   "8:5", 30.556, 30.556}, /* Mode 9  */
   &nv_ch1_pal_tiny_c,  &ch1_pal_tiny_c,  ch1_fsci_pal_tiny_c,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},    
  {{TV_SYSTEM_PAL, 640, 400, "Small",  "8:5", 13.194, 13.194}, /* Mode 8  */
   &nv_ch1_pal_small_c, &ch1_pal_small_c, ch1_fsci_pal_small_c,
   PORT_NVIDIA, TV_DEF_CH | TV_CAP_MACROVISION}, /* FIXME: No Dual view yet */
  {{TV_SYSTEM_PAL, 720, 400, "Tiny",   "9:5", 30.556, 30.556}, /* Mode 5  */
   &nv_ch1_pal_tiny_d,  &ch1_pal_tiny_d,  ch1_fsci_pal_tiny_d,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},    
  {{TV_SYSTEM_PAL, 720, 400, "Small",  "9:5", 13.194, 13.194}, /* Mode 4  */
   &nv_ch1_pal_small_d, &ch1_pal_small_d, ch1_fsci_pal_small_d,
   PORT_NVIDIA, TV_DEF_CH}, /* FIXME: No Dualview yet */
  {{TV_SYSTEM_PAL, 512, 384, "Tiny",   "4:3", 33.333, 33.333}, /* Mode 1  */
   &nv_ch1_pal_tiny_e,  &ch1_pal_tiny_e,  ch1_fsci_pal_tiny_e,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},    
  {{TV_SYSTEM_PAL, 512, 384, "Small",  "4:3", 16.667, 16.667}, /* Mode 0  */
   &nv_ch1_pal_small_e, &ch1_pal_small_e, ch1_fsci_pal_small_e,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},

  {{TV_SYSTEM_PAL,  768, 576, "Small",  "4:3", 0.000,  0.000}, /* Mode 20' */
   &nv_ch1_pal_small_f, &ch1_pal_small_f, ch1_fsci_pal_small_b,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},	      
  {{TV_SYSTEM_PAL,  768, 576, "Large",  "4:3", 0.000,  0.000}, /* Mode 19' */
   &nv_ch1_pal_large_f, &ch1_pal_large_f, ch1_fsci_pal_huge_b,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},	      
  {{TV_SYSTEM_PAL,  800, 450, "Small", "16:9", 0.000,  0.000}, /* Mode 20" */
   &nv_ch1_pal_small_g, &ch1_pal_small_g, ch1_fsci_pal_small_b,	      
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},	      
  {{TV_SYSTEM_PAL,  800, 450, "Large", "16:9", 0.000,  0.000}, /* Mode 19" */
   &nv_ch1_pal_large_g, &ch1_pal_large_g, ch1_fsci_pal_huge_b,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},

#ifdef CONFIG_SLAVE_CH
  {{TV_SYSTEM_PAL, 800, 600, "Slave$", "4:3", 13.194, 13.194}, /* Mode 20 */
   &nv_ch1_pal_slave_b, &ch1_pal_slave_b, ch1_fsci_pal_small_b,
   PORT_NVIDIA_SYNC_SLAVE, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},
  {{TV_SYSTEM_PAL, 640, 480, "Slave$", "4:3", 16.667, 16.667}, /* Mode 14 */
   &nv_ch1_pal_slave_a, &ch1_pal_small_a, ch1_fsci_pal_small_a,
   PORT_NVIDIA_SYNC_SLAVE, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},  
#endif

#if 0 /* not yet supported */
  {{TV_SYSTEM_PAL,  720, 576, "Interl", 0.000,  0.000},   /* Mode 25 */
   &nv_ch1_pal_interl_dvd, &ch1_pal_interl_dvd, NULL, PORT_NVIDIA, TV_DEF_CH},
#endif

  {{TV_SYSTEM_NONE, 0, 0, NULL, NULL, 0.0, 0.0}, NULL, NULL, NULL, 0},
};

/* NTSC modes with "no dot crawl" get # appended to size */

/* 60 Hz: NTSC, NTSC-NDC, PAL-M, PAL-60 */

TVTemplateNvCh templ_nv_ch1_60hz [] = { 
  {{TV_SYSTEM_NTSC, 800, 600, "Small",  "4:3", 12.500, 12.500}, /* Mode 24 */
   &nv_ch1_ntsc_small_b, &ch1_ntsc_small_b, ch1_fsci_ntsc_small_b,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},
  {{TV_SYSTEM_NTSC, 800, 600, "Large",  "4:3",  6.250,  6.250}, /* Mode 23 */
   &nv_ch1_ntsc_large_b, &ch1_ntsc_large_b, ch1_fsci_ntsc_large_b,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},
  {{TV_SYSTEM_NTSC, 800, 600, "Huge",   "4:3", -4.167, -4.167}, /* Mode 22 */
   &nv_ch1_ntsc_huge_b,  &ch1_ntsc_huge_b,  ch1_fsci_ntsc_huge_b,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},
  {{TV_SYSTEM_NTSC, 640, 480, "Tiny",   "4:3", 16.667, 16.667}, /* Mode 18 */
   &nv_ch1_ntsc_tiny_a,  &ch1_ntsc_tiny_a,  ch1_fsci_ntsc_tiny_a,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},
  {{TV_SYSTEM_NTSC, 640, 480, "Small",  "4:3", 12.500, 12.500}, /* Mode 17 */
   &nv_ch1_ntsc_small_a, &ch1_ntsc_small_a, ch1_fsci_ntsc_small_a,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},
  {{TV_SYSTEM_NTSC, 640, 480, "Huge",   "4:3",  0.000,  0.000}, /* Mode 16 */
   &nv_ch1_ntsc_huge_a, &ch1_ntsc_huge_a, ch1_fsci_ntsc_huge_a,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW}, 
  {{TV_SYSTEM_NTSC, 640, 400, "Tiny",   "8:5", 27.083, 27.083}, /* Mode 12 */
   &nv_ch1_ntsc_tiny_c,  &ch1_ntsc_tiny_c,  ch1_fsci_ntsc_tiny_c,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},	      
  {{TV_SYSTEM_NTSC, 640, 400, "Small",  "8:5", 16.667, 16.667}, /* Mode 11 */
   &nv_ch1_ntsc_small_c, &ch1_ntsc_small_c, ch1_fsci_ntsc_small_c,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},	      
  {{TV_SYSTEM_NTSC, 640, 400, "Huge",   "8:5", -4.167, -4.167}, /* Mode 10 */
   &nv_ch1_ntsc_huge_c, &ch1_ntsc_huge_c, ch1_fsci_ntsc_huge_c,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},	      
  {{TV_SYSTEM_NTSC, 720, 400, "Small",  "9:5", 16.667, 16.667}, /* Mode 7  */
   &nv_ch1_ntsc_small_d, &ch1_ntsc_small_d, ch1_fsci_ntsc_small_d,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},	      
  {{TV_SYSTEM_NTSC, 720, 400, "Huge",   "9:5", -4.167, -4.167}, /* Mode 6  */
   &nv_ch1_ntsc_huge_d, &ch1_ntsc_huge_d, ch1_fsci_ntsc_huge_d,
   PORT_NVIDIA, TV_DEF_CH}, /* FIXME: No Dualview yet */
  {{TV_SYSTEM_NTSC, 512, 384, "Small",  "4:3", 20.000, 20.000}, /* Mode 3  */
   &nv_ch1_ntsc_small_e, &ch1_ntsc_small_e, ch1_fsci_ntsc_small_e,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},
  {{TV_SYSTEM_NTSC, 512, 384, "Large",  "4:3",  0.000,  0.000}, /* Mode 2  */
   &nv_ch1_ntsc_large_e, &ch1_ntsc_large_e, ch1_fsci_ntsc_large_e,
   PORT_NVIDIA, TV_DEF_CH}, /* FIXME: No Dualview */

  {{TV_SYSTEM_NTSC,  768, 576, "Large",  "4:3", 0.000, 0.000},  /* Mode 23' */
   &nv_ch1_ntsc_large_f, &ch1_ntsc_large_f, ch1_fsci_ntsc_large_b,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},
  {{TV_SYSTEM_NTSC,  800, 450, "Large", "16:9", 0.000, 0.000},  /* Mode 23" */
   &nv_ch1_ntsc_large_g, &ch1_ntsc_large_g, ch1_fsci_ntsc_large_b,
   PORT_NVIDIA, TV_DEF_CH | TV_DEF_DUALVIEW},

#ifdef CONFIG_SLAVE_CH
  {{TV_SYSTEM_NTSC, 800, 600, "Slave$", "4:3", 12.500, 12.500}, /* Mode 24 */
   &nv_ch1_ntsc_slave_b, &ch1_ntsc_slave_b, ch1_fsci_ntsc_small_b,
   PORT_NVIDIA_SYNC_SLAVE, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},
  {{TV_SYSTEM_NTSC, 640, 480, "Slave$", "4:3", 12.500, 12.500}, /* Mode 17 */
   &nv_ch1_ntsc_slave_a, &ch1_ntsc_small_a, ch1_fsci_ntsc_small_a,
   PORT_NVIDIA_SYNC_SLAVE, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},
#endif

#if 0 /* not yet supported, maybe never will */
  {{TV_SYSTEM_NTSC,  720, 480, "Interl", "DVD", 0.000,  0.000}, /* Mode 26 */
   &nv_ch1_ntsc_interl_dvd, &ch1_ntsc_interl_dvd, NULL, PORT_NVIDIA, TV_DEF_CH},
#endif

  {{TV_SYSTEM_NONE, 0, 0, NULL, NULL, 0.0, 0.0}, NULL, NULL, NULL, 0},
};

/* -------- -------- */

static TVMode *modes_nv_ch1 = NULL;
static TVMode *modes_nv_ch2 = NULL;

static TVMode*
loop_create_nv_ch1 (TVMode *m, TVTemplateNvCh *t, 
  TVSystem system, int f, char *suffix)
{
  for (; t->spec.system != TV_SYSTEM_NONE; t++) {
    m->spec = t->spec;
    m->spec.system = system;
    m->descFlags = t->descFlags;
    if (suffix) {
      register char *s;

      s = (char *) malloc ((strlen(t->spec.size) + strlen (suffix) + 1) *
			   sizeof (char));
      m->spec.size = strcat (strcpy (s, t->spec.size), suffix);
    }
    m->regs.devFlags = (t->descFlags & TV_CAP_DUALVIEW) ? 
      (DEV_MONITOR | DEV_TELEVISION) : (DEV_TELEVISION);
    m->regs.portHost = m->regs.portEnc = t->portFlags;
    m->regs.crtc.nv = *(t->nv);
    m->regs.enc.ch = *(t->ch);
    if (t->fsci) m->regs.enc.ch.fsci = t->fsci[f];
    data_init_ch1 (system, &m->regs.enc.ch);
    data_init_nv (&m->regs.crtc.nv, m->regs.portHost, m->regs.devFlags);
    m++;
  }
  return m;
}

static TVMode*
loop_create_nv_ch2 (TVMode *m, TVTemplateNvCh *t, 
  TVSystem system, int f, char *suffix)
{
  for (; t->spec.system != TV_SYSTEM_NONE; t++) {
    m->spec = t->spec;
    m->spec.system = system;
    m->descFlags = t->descFlags;
    if (suffix) {
      register char *s;

      s = (char *) malloc ((strlen(t->spec.size) + strlen (suffix) + 1) *
			   sizeof (char));
      m->spec.size = strcat (strcpy (s, t->spec.size), suffix);
    }
    m->regs.devFlags = (t->descFlags & TV_CAP_DUALVIEW) ? 
      (DEV_MONITOR | DEV_TELEVISION) : (DEV_TELEVISION);
    m->regs.portHost = m->regs.portEnc = t->portFlags;
    m->regs.crtc.nv = *(t->nv);
    m->regs.enc.ch = *(t->ch);
    if (t->fsci) m->regs.enc.ch.fsci = t->fsci[f];
    data_init_ch2 (system, &m->regs.enc.ch);
    data_init_nv (&m->regs.crtc.nv, m->regs.portHost, m->regs.devFlags);
    m++;
  }
  return m;
}


TVMode*
data_modes_nv_ch1 (void)
{
  int c;
  TVMode *m;

  if (modes_nv_ch1) return modes_nv_ch1;
  c = sizeof (templ_nv_ch1_60hz) / sizeof(TVTemplateNvCh) * 5
    + sizeof (templ_nv_ch1_50hz) / sizeof(TVTemplateNvCh) * 3
    + 1;
  modes_nv_ch1 = (TVMode *) malloc (sizeof (TVMode) * c);
  
  m = modes_nv_ch1;
  m = loop_create_nv_ch1 (m, templ_nv_ch1_60hz, TV_SYSTEM_NTSC,   0, NULL);
  m = loop_create_nv_ch1 (m, templ_nv_ch1_60hz, TV_SYSTEM_NTSC_J, 0, NULL);
  m = loop_create_nv_ch1 (m, templ_nv_ch1_60hz, TV_SYSTEM_NTSC,   1, "#");
  m = loop_create_nv_ch1 (m, templ_nv_ch1_60hz, TV_SYSTEM_PAL_M,  2, NULL);
  m = loop_create_nv_ch1 (m, templ_nv_ch1_60hz, TV_SYSTEM_PAL_60, 3, NULL);
  m = loop_create_nv_ch1 (m, templ_nv_ch1_50hz, TV_SYSTEM_PAL,    0, NULL);
  m = loop_create_nv_ch1 (m, templ_nv_ch1_50hz, TV_SYSTEM_PAL_N,  1, NULL);
  m = loop_create_nv_ch1 (m, templ_nv_ch1_50hz, TV_SYSTEM_PAL_X,  2, NULL);
  m->spec.system = TV_SYSTEM_NONE;
  return modes_nv_ch1;
}

TVMode*
data_modes_nv_ch2 (void)
{
  int c;
  TVMode *m;

  c = 0
    + 1;
  modes_nv_ch2 = (TVMode *) malloc (sizeof (TVMode) * c);
  
  m = modes_nv_ch2;
  m->spec.system = TV_SYSTEM_NONE;
  return modes_nv_ch2;
}

/* ======== PH ================================ */

#define TV_DEF_PH (TV_CAP_MACROVISION | TV_CAP_NONINTERLACED \
                   | TV_CAP_DUALVIEW | TV_CAP_COLORFIX)

/* -------- PH -------- NTSC -------- */

/* -------- 640x480 -------- */

TVNvRegs nv_ph_ntsc_small_a = { 
  HDisplay    : 640,
  HSyncStart  : 656,
  HSyncEnd    : 752,
  HTotal      : 800, /* verified */
  VDisplay    : 480,
  VSyncStart  : 536, /* test! */
  VSyncEnd    : 538, /* test! */
  VTotal      : 620, /* test! */
  flags       : 0,
};

TVNvRegs nv_ph_ntsc_medium_a = { 
  HDisplay    : 640,
  HSyncStart  : 656,
  HSyncEnd    : 752,
  HTotal      : 800, /* verified */
  VDisplay    : 480,
  VSyncStart  : 519, /* test! */
  VSyncEnd    : 521, /* test! */
  VTotal      : 585, /* test! */
  flags       : 0,
};

/* -------- 800x600 -------- */

TVNvRegs nv_ph_ntsc_small_b = { 
  HDisplay    : 800,
  HSyncStart  : 824,  /* VESA 56Hz */
  HSyncEnd    : 896,  /* VESA 56Hz */
  HTotal      : 1024, /* verified */
  VDisplay    : 600,
  VSyncStart  : 668, /* test! */
  VSyncEnd    : 670, /* test! */
  VTotal      : 774, /* test! */ 
  flags       : 0,
};

TVNvRegs nv_ph_ntsc_medium_b = { 
  HDisplay    : 800,
  HSyncStart  : 824,  /* VESA 56Hz */
  HSyncEnd    : 896,  /* VESA 56Hz */
  HTotal      : 1024, /* verified */
  VDisplay    : 600,
  VSyncStart  : 649, /* test! */
  VSyncEnd    : 651, /* test! */
  VTotal      : 731, /* test! */  
  flags       : 0,
};

TVNvRegs nv_ph_ntsc_huge_b = { 
  HDisplay    : 800,
  HSyncStart  : 824,  /* VESA 56Hz */
  HSyncEnd    : 896,  /* VESA 56Hz */
  HTotal      : 1024, /* verified */
  VDisplay    : 600,
  VSyncStart  : 613, /* test! */
  VSyncEnd    : 615, /* test! */
  VTotal      : 655, /* test! */ 
  flags       : 0,
};

/* -------- PH -------- PAL -------- */

/* -------- 640x480 -------- */

TVNvRegs nv_ph_pal_small_a = { 
  HDisplay    : 640,
  HSyncStart  : 656,
  HSyncEnd    : 752,
  HTotal      : 800, /* verified */
  VDisplay    : 480,
  VSyncStart  : 526, /* test! */
  VSyncEnd    : 528, /* test! */
  VTotal      : 602, /* test! */
  flags       : 0,
};

TVNvRegs nv_ph_pal_medium_a = { 
  HDisplay    : 640,
  HSyncStart  : 656,
  HSyncEnd    : 752,
  HTotal      : 800, /* verified */
  VDisplay    : 480,
  VSyncStart  : 515, /* test! */
  VSyncEnd    : 517, /* test! */
  VTotal      : 581, /* test! */
  flags       : 0,
};

/* -------- 800x600 -------- */

TVNvRegs nv_ph_pal_small_b = { 
  HDisplay    : 800,
  HSyncStart  : 824,  /* VESA 56Hz */
  HSyncEnd    : 896,  /* VESA 56Hz */
  HTotal      : 1024, /* verified */
  VDisplay    : 600,
  VSyncStart  : 657, /* test! */
  VSyncEnd    : 659, /* test! */
  VTotal      : 752, /* test! */  
  flags       : 0,
};

TVNvRegs nv_ph_pal_medium_b = { 
  HDisplay    : 800,
  HSyncStart  : 824,  /* VESA 56Hz */
  HSyncEnd    : 896,  /* VESA 56Hz */
  HTotal      : 1024, /* verified */
  VDisplay    : 600,
  VSyncStart  : 643, /* test! */
  VSyncEnd    : 645, /* test! */
  VTotal      : 726, /* test! */  
  flags       : 0,
};

TVNvRegs nv_ph_pal_huge_b = { 
  HDisplay    : 800,
  HSyncStart  : 824,  /* VESA 56Hz */
  HSyncEnd    : 896,  /* VESA 56Hz */
  HTotal      : 1024, /* verified */
  VDisplay    : 600,
  VSyncStart  : 610, /* test! */
  VSyncEnd    : 612, /* test! */
  VTotal      : 653, /* test! */  
  flags       : 0,
};

/* ======== PH ======== SAA7104 ======================== */

/* -------- SAA7104 Test Master modes -------- */

TVNvRegs nv_ph_ntsc_test_a = { 
  HDisplay    : 640,
  HSyncStart  : 656,
  HSyncEnd    : 720,
  HTotal      : 768, 
  VDisplay    : 480,
  VSyncStart  : 526, 
  VSyncEnd    : 528, 
  VTotal      : 602, 
  flags       : 0,
};

TVNvRegs nv_ph_ntsc_test_b = { 
  HDisplay    : 800,
  HSyncStart  : 928,  
  HSyncEnd    : 1000,
  HTotal      : 1216, 
  VDisplay    : 600,
  VSyncStart  : 657, 
  VSyncEnd    : 659, 
  VTotal      : 752, 
  flags       : 0,
};

TVNvRegs nv_ph_ntsc_test_c = { /* Master/Slave */
  /* Clock 2*32.799 MHz -> 1140x960 -> 59.94 Hz ok */
  HDisplay    : 1024,
  HSyncStart  : 1072,
  HSyncEnd    : 1128,
  HTotal      : 1140, /* rem 4 */
  VDisplay    : 768,
  VSyncStart  : 820, 
  VSyncEnd    : 822, 
  VTotal      : 960, 
  flags       : 0,
  slave : {
    HTotal     : 1140,
    HSyncStart : 1,
    HSyncEnd   : 64,
    VTotal     : 960,
    VSyncStart : 1,
    VSyncEnd   : 16,
    Unknown    : 6,
  }
};

/* ---- */

TVNvRegs nv_ph_pal_test_a = { 
  HDisplay    : 640,
  HSyncStart  : 656,
  HSyncEnd    : 752,
  HTotal      : 808, 
  VDisplay    : 480,
  VSyncStart  : 526, 
  VSyncEnd    : 528, 
  VTotal      : 602, 
  flags       : 0,
};

TVNvRegs nv_ph_pal_test_b = { 
  HDisplay    : 800,
  HSyncStart  : 824,  /* VESA 56Hz */
  HSyncEnd    : 896,  /* VESA 56Hz */
  HTotal      : 1032, 
  VDisplay    : 600,
  VSyncStart  : 657, 
  VSyncEnd    : 659, 
  VTotal      : 752, 
  flags       : 0,
};

TVNvRegs nv_ph_pal_test_c = { /* Master/Slave */
  /* Clock 2*28.500 MHz -> 1200x950 -> 50 Hz ok */
  HDisplay    : 1024,
  HSyncStart  : 1072,
  HSyncEnd    : 1128,
  HTotal      : 1200, 
  VDisplay    : 768,
  VSyncStart  : 820, 
  VSyncEnd    : 822, 
  VTotal      : 950, 
  flags       : 0,
  slave : {
    HTotal     : 1200,
    HSyncStart : 1,
    HSyncEnd   : 64,
    VTotal     : 950,
    VSyncStart : 1,
    VSyncEnd   : 16,
    Unknown    : 6,
  }
};

/* -------- SAA7104 Test Slave modes -------- */

TVNvRegs nv_ph_ntsc_slave_a = { /* 640x480 Slave */
  /* Clock 28.288 MHz -> 800x590 -> 59.93 Hz ok */
  HDisplay    :	640,
  HSyncStart  :	720,
  HSyncEnd    :	744,
  HTotal      :	800,
  VDisplay    :	480,
  VSyncStart  :	520,
  VSyncEnd    :	522,
  VTotal      :	590,
  slave : {
    HTotal     : 800,
    HSyncStart : 1,
    HSyncEnd   : 64,
    VTotal     : 590,
    VSyncStart : 1,
    VSyncEnd   : 16,
    Unknown    : 0,
  }
};

TVNvRegs nv_ph_ntsc_slave_b = { /* 800x600 Slave */
  /* Clock 47.832 MHz -> 1064x750 -> 59.94 Hz ok */
  HDisplay    :	800,
  HSyncStart  :	964,
  HSyncEnd    :	1028,
  HTotal      :	1064,
  VDisplay    :	600,
  VSyncStart  :	651,
  VSyncEnd    :	654,
  VTotal      :	750,
  slave : {
    HTotal     : 1064,
    HSyncStart : 1,
    HSyncEnd   : 64,
    VTotal     : 750,
    VSyncStart : 1,
    VSyncEnd   : 16,
    Unknown    : 0,
  }
};

/* ---- */

TVNvRegs nv_ph_pal_slave_a = { /* 640x480 Slave */
  /* Clock 25.197 -> 840x600 -> 49.99 Hz */
  HDisplay    :	640,
  HSyncStart  :	736,
  HSyncEnd    :	752,
  HTotal      :	832,
  VDisplay    :	480,
  VSyncStart  :	520, /* 480 */
  VSyncEnd    :	522, /* 484 */
  VTotal      :	600, /* 304 ?? */
  slave : {
    HTotal     : 840,
    HSyncStart : 1,
    HSyncEnd   : 64,
    VTotal     : 600,
    VSyncStart : 1,
    VSyncEnd   : 16,
    Unknown    : 12,
  }
};

TVNvRegs nv_ph_pal_slave_b = { /* 800x600 Slave */
  /* Clock 36.000 -> 1000x720 -> 50 Hz ok */
  HDisplay    :	800,
  HSyncStart  :	992,
  HSyncEnd    :	1152,
  HTotal      :	1056,
  VDisplay    :	600,
  VSyncStart  :	601,
  VSyncEnd    :	604,
  VTotal      :	694,
  slave : {
    HTotal     : 1000,
    HSyncStart : 1,
    HSyncEnd   : 64,
    VTotal     : 720,
    VSyncStart : 1,
    VSyncEnd   : 16,
    Unknown    : 6,
  }
};

/* -------- PH modes -------- */

typedef struct {
  TVModeSpec spec;
  TVNvRegs *nv;
  TVPh1Regs *ph;
  int portFlags;
  int descFlags;
} TVTemplateNvPh1;

typedef struct {
  TVModeSpec spec;
  TVNvRegs *nv;
  TVPh2Regs *ph;
  int portFlags;
  int descFlags;
} TVTemplateNvPh2;

TVTemplateNvPh1 templ_nv_ph1 [] = {

  {{TV_SYSTEM_PAL, 640, 480, "Small",  "4:3", 11.68, 13.19}, 
   &nv_ph_pal_small_a,  &ph_pal_small_a,  PORT_NVIDIA, TV_DEF_PH},
  {{TV_SYSTEM_PAL, 640, 480, "Medium", "4:3", 08.83, 10.07}, 
   &nv_ph_pal_medium_a, &ph_pal_medium_a, PORT_NVIDIA, TV_DEF_PH},

  {{TV_SYSTEM_PAL, 800, 600, "Small",  "4:3", 11.68, 13.19}, 
   &nv_ph_pal_small_b,  &ph_pal_small_b,  PORT_NVIDIA, TV_DEF_PH},
  {{TV_SYSTEM_PAL, 800, 600, "Medium", "4:3", 08.83, 10.07}, 
   &nv_ph_pal_medium_b, &ph_pal_medium_b, PORT_NVIDIA, TV_DEF_PH},
  {{TV_SYSTEM_PAL, 800, 600, "Huge",   "4:3", 00.00, 00.00}, 
   &nv_ph_pal_huge_b,   &ph_pal_huge_b,   PORT_NVIDIA, TV_DEF_PH},

  {{TV_SYSTEM_NTSC, 640, 480, "Small",  "4:3", 12.67, 16.05},
   &nv_ph_ntsc_small_a,  &ph_ntsc_small_a,  PORT_NVIDIA, TV_DEF_PH},
  {{TV_SYSTEM_NTSC, 640, 480, "Medium", "4:3", 09.86, 11.11},
   &nv_ph_ntsc_medium_a, &ph_ntsc_medium_a, PORT_NVIDIA, TV_DEF_PH},

  {{TV_SYSTEM_NTSC, 800, 600, "Small",  "4:3", 12.67, 16.05},
   &nv_ph_ntsc_small_b,  &ph_ntsc_small_b,  PORT_NVIDIA, TV_DEF_PH},
  {{TV_SYSTEM_NTSC, 800, 600, "Medium", "4:3", 09.86, 11.11}, 
   &nv_ph_ntsc_medium_b, &ph_ntsc_medium_b, PORT_NVIDIA, TV_DEF_PH},
  {{TV_SYSTEM_NTSC, 800, 600, "Huge",   "4:3", 00.00, 00.82}, 
   &nv_ph_ntsc_huge_b,   &ph_ntsc_huge_b,   PORT_NVIDIA, TV_DEF_PH},

  {{TV_SYSTEM_NONE, 0, 0, NULL, NULL, 0.0, 0.0}, NULL, NULL, 0},
};

TVTemplateNvPh2 templ_nv_ph2 [] = {
  {{TV_SYSTEM_PAL,  640, 480, "Slave$", "4:3", 00.00, 00.00}, 
   &nv_ph_pal_slave_a, &ph_pal_slave_a, PORT_NVIDIA_SYNC_SLAVE, TV_DEF_PH},
  {{TV_SYSTEM_PAL,  800, 600, "Slave$", "4:3", 00.00, 00.00}, 
   &nv_ph_pal_slave_b, &ph_pal_slave_b, PORT_NVIDIA_SYNC_SLAVE, TV_DEF_PH},
  {{TV_SYSTEM_PAL, 1024, 768, "Slave$", "4:3", 00.00, 00.00}, 
   &nv_ph_pal_test_c,  &ph_pal_test_c,  PORT_NVIDIA_SYNC_SLAVE, TV_DEF_PH},

  {{TV_SYSTEM_NTSC,  640, 480, "Slave$", "4:3", 00.00, 00.00}, 
   &nv_ph_ntsc_slave_a, &ph_ntsc_slave_a, PORT_NVIDIA_SYNC_SLAVE, TV_DEF_PH},
  {{TV_SYSTEM_NTSC,  800, 600, "Slave$", "4:3", 00.00, 00.00}, 
   &nv_ph_ntsc_slave_b, &ph_ntsc_slave_b, PORT_NVIDIA_SYNC_SLAVE, TV_DEF_PH},
  {{TV_SYSTEM_NTSC, 1024, 768, "Slave$", "4:3", 00.00, 00.00}, 
   &nv_ph_ntsc_test_c,  &ph_ntsc_test_c,  PORT_NVIDIA_SYNC_SLAVE, TV_DEF_PH},

  {{TV_SYSTEM_PAL,  640, 480, "Test", "4:3", 00.00, 00.00}, 
   &nv_ph_pal_test_a,  &ph_pal_test_a,  PORT_NVIDIA, TV_DEF_PH},
  {{TV_SYSTEM_PAL,  800, 600, "Test", "4:3", 00.00, 00.00}, 
   &nv_ph_pal_test_b,  &ph_pal_test_b,  PORT_NVIDIA, TV_DEF_PH},
  {{TV_SYSTEM_PAL, 1024, 768, "Test", "4:3", 00.00, 00.00}, 
   &nv_ph_pal_test_c,  &ph_pal_test_c,  PORT_NVIDIA, TV_DEF_PH},

  {{TV_SYSTEM_NTSC,  640, 480, "Test", "4:3", 00.00, 00.00}, 
   &nv_ph_ntsc_test_a,  &ph_ntsc_test_a,  PORT_NVIDIA, TV_DEF_PH},
  {{TV_SYSTEM_NTSC,  800, 600, "Test", "4:3", 00.00, 00.00}, 
   &nv_ph_ntsc_test_b,  &ph_ntsc_test_b,  PORT_NVIDIA, TV_DEF_PH},
  {{TV_SYSTEM_NTSC, 1024, 768, "Test", "4:3", 00.00, 00.00}, 
   &nv_ph_ntsc_test_c,  &ph_ntsc_test_c,  PORT_NVIDIA, TV_DEF_PH},

  {{TV_SYSTEM_NONE, 0, 0, NULL, NULL, 0.0, 0.0}, NULL, NULL, 0},
};


/* -------- -------- */

static TVMode *modes_nv_ph1 = NULL;
static TVMode *modes_nv_ph2 = NULL;

/*
 * count number of modes matching system; TV_SYSTEM_NONE matches all.
 */

int 
data_count_nv_ph1 (TVSystem system)
{
  int count;
  TVTemplateNvPh1 *t;

  count = 0;
  for (t = templ_nv_ph1; t->spec.system != TV_SYSTEM_NONE; t++) {
    if (system == TV_SYSTEM_NONE || system == t->spec.system) count++;
  }
  return count;
}

int 
data_count_nv_ph2 (TVSystem system)
{
  int count;
  TVTemplateNvPh2 *t;

  count = 0;
  for (t = templ_nv_ph2; t->spec.system != TV_SYSTEM_NONE; t++) {
    if (system == TV_SYSTEM_NONE || system == t->spec.system) count++;
  }
  return count;
}

TVMode*
data_modes_nv_ph1 (void)
{
  int c;
  TVMode *m;
  TVTemplateNvPh1 *t;

  if (modes_nv_ph1) return modes_nv_ph1;
  c = data_count_nv_ph1 (TV_SYSTEM_NONE) + 1;    
  modes_nv_ph1 = (TVMode *) malloc (sizeof (TVMode) * c);
  
  m = modes_nv_ph1;
  for (t = templ_nv_ph1; t->spec.system != TV_SYSTEM_NONE; t++) {
    m->spec = t->spec;
    m->descFlags = t->descFlags;
    m->regs.portHost = m->regs.portEnc = t->portFlags;
    m->regs.devFlags = (t->descFlags & TV_CAP_DUALVIEW) ? 
      (DEV_MONITOR | DEV_TELEVISION) : (DEV_TELEVISION);
    m->regs.crtc.nv = *(t->nv);
    m->regs.enc.ph1 = *(t->ph);
    data_init_ph1 (m->spec.system, &m->regs.enc.ph1);
    data_init_nv (&m->regs.crtc.nv, m->regs.portHost, m->regs.devFlags);
    m++;
  }
  m->spec.system = TV_SYSTEM_NONE;
  return modes_nv_ph1;
}

TVMode*
data_modes_nv_ph2 (void)
{
  int c;
  TVMode *m;
  TVTemplateNvPh1 *t1;
  TVTemplateNvPh2 *t2;

  if (modes_nv_ph2) return modes_nv_ph2;
  c = data_count_nv_ph1 (TV_SYSTEM_NONE) +
      data_count_nv_ph2 (TV_SYSTEM_NONE) + 1;    
  modes_nv_ph2 = (TVMode *) malloc (sizeof (TVMode) * c);
  
  m = modes_nv_ph2;
  for (t1 = templ_nv_ph1; t1->spec.system != TV_SYSTEM_NONE; t1++) {
    m->spec = t1->spec;
    m->descFlags = t1->descFlags;
    m->regs.portHost = m->regs.portEnc = t1->portFlags;
    m->regs.devFlags = (t1->descFlags & TV_CAP_DUALVIEW) ? 
      (DEV_MONITOR | DEV_TELEVISION) : (DEV_TELEVISION);
    m->regs.crtc.nv = *(t1->nv);
    m->regs.enc.ph1 = *(t1->ph);
    m->regs.enc.ph2.pcle = 1;
    m->regs.enc.ph2.pcli = 1;
    data_init_ph2 (m->spec.system, &m->regs.enc.ph2);
    data_init_nv (&m->regs.crtc.nv, m->regs.portHost, m->regs.devFlags);
    m++;
  }
  for (t2 = templ_nv_ph2; t2->spec.system != TV_SYSTEM_NONE; t2++) {
    m->spec = t2->spec;
    m->descFlags = t2->descFlags;
    m->regs.portHost = m->regs.portEnc = t2->portFlags;
    m->regs.devFlags = (t2->descFlags & TV_CAP_DUALVIEW) ? 
      (DEV_MONITOR | DEV_TELEVISION) : (DEV_TELEVISION);
    m->regs.crtc.nv = *(t2->nv);
    m->regs.enc.ph2 = *(t2->ph);
    data_init_ph2 (m->spec.system, &m->regs.enc.ph2);
    data_init_nv (&m->regs.crtc.nv, m->regs.portHost, m->regs.devFlags);
    m++;
  }
  m->spec.system = TV_SYSTEM_NONE;
  return modes_nv_ph2;
}

/* ======== NX ======== NVIDIA Internal ======================== */

#define TV_DEF_NX 0

/* -------- CRT -------- */

TVNvRegs nv_nx_normal_a = { /* All 640x480 */
  HDisplay    : 640,
  HSyncStart  : 656,
  HSyncEnd    : 752,
  HTotal      : 880,
  HBlankStart : 640,
  HBlankEnd   : 880,
  VDisplay    : 480,
  VSyncStart  : 480,
  VSyncEnd    : 492,
  VTotal      : 525,
  VBlankStart : 480,
  VBlankEnd   : 525,
  flags       : 0,
  clock       : 0, 
};

TVNvRegs nv_nx_normal_b = { /* All 800x600 */
  HDisplay    : 800,
  HSyncStart  : 840,
  HSyncEnd    : 968,
  HTotal      : 1056,
  HBlankStart : 800,
  HBlankEnd   : 1056,
  VDisplay    : 600,
  VSyncStart  : 600,
  VSyncEnd    : 605,
  VTotal      : 628,
  VBlankStart : 600,
  VBlankEnd   : 628,
  flags       : 0,
  clock       : 0, 
};

TVNvRegs nv_nx_normal_c = { /* NTSC 1024x768 */
  HDisplay    : 1024,
  HSyncStart  : 1048,
  HSyncEnd    : 1184,
  HTotal      : 1344,
  HBlankStart : 1024,
  HBlankEnd   : 1344,
  VDisplay    : 768,
  VSyncStart  : 768,
  VSyncEnd    : 777,
  VTotal      : 806,
  VBlankStart : 768,
  VBlankEnd   : 806,
  flags       : 0,
  clock       : 0, 
};

/* -------- NX modes -------- */

typedef struct {
  TVModeSpec spec;
  TVNvRegs *nv;
  TVNxImgRegs *nx;
  int portFlags;
  int descFlags;
} TVTemplateNvNx;

TVTemplateNvNx templ_nv_nx_60hz [] = {
  {{TV_SYSTEM_NTSC,  640, 480, "Normal", "4:3", 08.75, 08.75}, 
   &nv_nx_normal_a, &nx_60_normal_a, PORT_NVIDIA, TV_DEF_NX},
  {{TV_SYSTEM_NTSC,  800, 600, "Normal", "4:3", 08.75, 08.75}, 
   &nv_nx_normal_b, &nx_60_normal_b, PORT_NVIDIA, TV_DEF_NX},
  {{TV_SYSTEM_NTSC, 1024, 768, "Normal", "4:3", 08.75, 08.75}, 
   &nv_nx_normal_c, &nx_60_normal_c, PORT_NVIDIA, TV_DEF_NX},

  {{TV_SYSTEM_NONE, 0, 0, NULL, NULL, 0.0, 0.0}, NULL, NULL, 0},
};

TVTemplateNvNx templ_nv_nx_50hz [] = {
  {{TV_SYSTEM_PAL, 640, 480, "Normal", "4:3", 08.75, 08.75},
   &nv_nx_normal_a, &nx_50_normal_a, PORT_NVIDIA, TV_DEF_NX},
  {{TV_SYSTEM_PAL, 800, 600, "Normal", "4:3", 08.75, 08.75},
   &nv_nx_normal_b, &nx_50_normal_b, PORT_NVIDIA, TV_DEF_NX},
  {{TV_SYSTEM_PAL, 1024, 768, "Normal", "4:3", 08.75, 08.75}, 
   &nv_nx_normal_c, &nx_50_normal_c, PORT_NVIDIA, TV_DEF_NX},

  {{TV_SYSTEM_NONE, 0, 0, NULL, NULL, 0.0, 0.0}, NULL, NULL, 0},
};

/* -------- -------- */

static TVMode *modes_nv_nx = NULL;

static TVMode*
loop_create_nv_nx (TVMode *m, TVTemplateNvNx *t, TVSystem system)
{
  for (; t->spec.system != TV_SYSTEM_NONE; t++) {
    m->spec = t->spec;
    m->spec.system = system;
    m->descFlags = t->descFlags;
    m->regs.devFlags = DEV_TELEVISION | DEV_INTERNAL;
    m->regs.portHost = m->regs.portEnc = t->portFlags;
    m->regs.crtc.nv = *(t->nv);
    m->regs.enc.nx.img = *(t->nx);
    data_init_nx (system, &m->regs.enc.nx); /* initializes rest */
    data_init_nv (&m->regs.crtc.nv, m->regs.portHost, m->regs.devFlags);
    data_complete_nx (&m->regs);
    m++;
  }
  return m;
}

TVMode*
data_modes_nv_nx (void)
{
  int c;
  TVMode *m;

  if (modes_nv_nx) return modes_nv_nx;
  c = sizeof (templ_nv_nx_60hz) / sizeof(TVTemplateNvNx) * 3
    + sizeof (templ_nv_nx_50hz) / sizeof(TVTemplateNvNx) * 3
    + 1;
  modes_nv_nx = (TVMode *) malloc (sizeof (TVMode) * c);
  
  m = modes_nv_nx;
  m = loop_create_nv_nx (m, templ_nv_nx_60hz, TV_SYSTEM_NTSC  );
  m = loop_create_nv_nx (m, templ_nv_nx_60hz, TV_SYSTEM_NTSC_J);
  m = loop_create_nv_nx (m, templ_nv_nx_60hz, TV_SYSTEM_PAL_M );
  m = loop_create_nv_nx (m, templ_nv_nx_50hz, TV_SYSTEM_PAL   );
  m = loop_create_nv_nx (m, templ_nv_nx_50hz, TV_SYSTEM_PAL_N );
  m = loop_create_nv_nx (m, templ_nv_nx_50hz, TV_SYSTEM_PAL_NC);
  m->spec.system = TV_SYSTEM_NONE;
  return modes_nv_nx;
}

/* ======== ================================ */

void data_init_nv (TVNvRegs *r, int portHost, int devFlags)
{
  if (devFlags & DEV_OVERLAY) {
    r->clock = 0;
    r->HBlankStart = 0;
    r->HBlankEnd   = 0;
    r->HTotal      = 0;
    r->VBlankStart = 0;
    r->VBlankEnd   = 0;
    r->VTotal      = 0;
  } else {
    r->HBlankStart = r->HDisplay;
    r->HBlankEnd   = r->HTotal;
    r->VBlankStart = r->VDisplay;
    r->VBlankEnd   = r->VTotal;
  }
  r->latency = 0;
  if ((portHost & PORT_SYNC_DIR) == PORT_SYNC_MASTER) 
  {
    r->slave.HSyncStart = 0;
    r->slave.HSyncEnd = 0;
    r->slave.HTotal = 0;
    r->slave.VSyncStart = 0;
    r->slave.VSyncEnd = 0;
    r->slave.VTotal = 0;
    r->slave.Unknown = 0;
  }
  r->flags = 0; /* FIXME */
}

void data_move_bt_cx (TVSettings *s, TVRegs *r)
{
  /* FIXME: Pure monitor movement, and tv vert movement */
  /* If HTotal = h_clki - 3*8, the monitor position is the same as at hclk_i.
     However, the image moves, so compensate with hsyncoffset. Try to keep
     hsynoffset always positive */
  /* Can't make HTotal too small, or monitor image will disappear... 
     So sometimes you have to use negative hsynoffset ... */
#if 0 /* TODO */
  if (r->crtc.HTotal > r->enc.bt.h_clki - 3*8 && 
      r->crtc.HTotal < r->enc.bt.h_clki) 
  {
    /* compensate for bad HTotal value */
    r->enc.bt.hsynoffset += r->crtc.HTotal - (r->enc.bt.h_clki - 3*8);
    r->crtc.HTotal = r->enc.bt.h_clki - 3*8;
  }
#endif
  r->crtc.nv.VSyncStart -= s->mon_voffset;
  r->crtc.nv.VSyncEnd   -= s->mon_voffset;
  r->crtc.nv.HSyncStart -= s->mon_hoffset * 8;
  r->crtc.nv.HSyncEnd   -= s->mon_hoffset * 8;
}

void data_move_nv (TVSettings *s, TVRegs *r)
{
  /* FIXME: Pure monitor movement */
  r->crtc.nv.VSyncStart -= s->mon_voffset;
  r->crtc.nv.VSyncEnd   -= s->mon_voffset;
  r->crtc.nv.HSyncStart -= s->mon_hoffset * 8;
  r->crtc.nv.HSyncEnd   -= s->mon_hoffset * 8;
}

void data_make_nv (int hdisplay, int hsyncstart, int hsyncend, 
  int htotal, int vdisplay, int vsyncstart, int vsyncend, int vtotal, 
  int dotclock, TVCrtcRegs *crt)
{
  crt->nv.clock      = dotclock;
  crt->nv.HDisplay   = hdisplay;  
  crt->nv.HSyncStart = hsyncstart;
  crt->nv.HSyncEnd   = hsyncend;  
  crt->nv.HTotal     = htotal;    
  crt->nv.VDisplay   = vdisplay;
  crt->nv.VSyncStart = vsyncstart;
  crt->nv.VSyncEnd   = vsyncend;  
  crt->nv.VTotal     = vtotal;    
  data_init_nv (&crt->nv, PORT_NVIDIA, DEV_MONITOR);
}

/* -------- -------- */

void data_calc_nv_bt (TVSystem system, int hres, int vres, 
  double hoc, double voc, TVRegs *r)
{
  recalc_bt_custom (system, hres, vres, hoc, voc, r);
  calc_nv_btcx (hres, vres, r); 
  data_mux_nv_bt (&r->enc.bt);
  data_init_nomux_bt (system, &r->enc.bt);
  r->portHost = r->portEnc = PORT_NVIDIA;
}

void data_calc_nv_cx (TVSystem system, int hres, int vres, 
  double hoc, double voc, TVRegs *r)
{
  recalc_cx_custom (system, hres, vres, hoc, voc, r);
  calc_nv_btcx (hres, vres, r); 
  data_init_cx (system, &r->enc.cx, 0);
  r->portHost = r->portEnc = PORT_NVIDIA;
}

/* -------- -------- */

void data_setup_nv_bt (TVSettings *s, TVRegs *r)
{
  data_move_nv (s, r); 
  data_move_bt_cx (s, r); 
  data_mux_nv_bt (&r->enc.bt);
  data_setup_nomux_bt (s, r);
}

void data_setup_nv_cx (TVSettings *s, TVRegs *r)
{
  data_move_nv (s, r); 
  data_move_bt_cx (s, r); 
  data_mux_nv_bt (&r->enc.bt);
  data_setup_cx (s, r);
}

void data_setup_nv_ch1 (TVSettings *s, TVRegs *r)
{
  data_move_nv (s, r); 
  data_setup_ch1 (s, r); 
}

void data_setup_nv_ch2 (TVSettings *s, TVRegs *r)
{
  data_move_nv (s, r); 
  data_setup_ch2 (s, r); 
}

void data_setup_nv_ph (TVSettings *s, TVRegs *r)
{
  data_move_nv (s, r); 
  data_setup_ph (s, r); 
}

void data_setup_nv_nx (TVSettings *s, TVRegs *r)
{
  data_move_nv (s, r); 
  data_setup_nx (s, r); 
}

/* -------- -------- */

DataCardFunc data_nv_func = {
  make: data_make_nv,
};

DataFunc data_nv_bt_func = {
  modes: data_modes_nv_bt,
  defaults: data_default_bt,
  setup: data_setup_nv_bt, 
  clamp: data_clamp_bt,
  calc:  data_calc_nv_bt,
};

DataFunc data_nv_cx_func = {
  modes: data_modes_nv_cx,
  defaults: data_default_cx,
  setup: data_setup_nv_cx, 
  clamp: data_clamp_cx,
  calc:  data_calc_nv_cx,
};

DataFunc data_nv_ch1_func = {
  modes: data_modes_nv_ch1,
  defaults: data_default_ch,
  setup: data_setup_nv_ch1, 
  clamp: data_clamp_ch,
  calc:  data_calc_null,
};

DataFunc data_nv_ch2_func = { /* FIXME */
  modes: data_modes_nv_ch2, 
  defaults: data_default_ch,
  setup: data_setup_nv_ch2, 
  clamp: data_clamp_ch,
  calc:  data_calc_null,
};

DataFunc data_nv_ph1_func = {
  modes: data_modes_nv_ph1,
  defaults: data_default_ph,
  setup: data_setup_nv_ph, 
  clamp: data_clamp_ph,
  calc:  data_calc_null,
};

DataFunc data_nv_ph2_func = {
  modes: data_modes_nv_ph2,
  defaults: data_default_ph,
  setup: data_setup_nv_ph, 
  clamp: data_clamp_ph,
  calc:  data_calc_null,
};

DataFunc data_nv_nx_func = {
  modes: data_modes_nv_nx,
  defaults: data_default_nx,
  setup: data_setup_nv_nx, 
  clamp: data_clamp_nx,
  calc:  data_calc_null,
};

